<?php

namespace Drupal\page_proxy;

use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\Routing\Route;

/**
 * Manager of page proxies.
 */
class PageProxyManager {

  /**
   * The page proxies.
   *
   * @var array
   */
  protected $pageProxies = [];

  /**
   * The routes of the page proxies.
   *
   * @var \Symfony\Component\Routing\Route[]
   */
  protected $routes = NULL;

  /**
   * Constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   A config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->pageProxies = [];
    foreach ($config_factory->get('page_proxy.settings')->get('page_proxies') ?? [] as $p) {
      $pid = \strtolower($p['path']);
      $pid = \preg_replace('/[^a-z0-9_]/', '_', $pid);
      $c = '';
      while (isset($this->pageProxies[$pid . $c])) {
        if ($c === '') {
          $c = 1;
        }
        else {
          $c += 1;
        }
      }
      $pid = $pid . $c;
      $this->pageProxies[$pid] = $p;
    }
  }

  /**
   * Callback for dynamic routes of page proxies.
   *
   * @return \Symfony\Component\Routing\Route[]
   *   Routes of configured page proxies.
   */
  public function routes() {
    if ($this->routes === NULL) {
      foreach ($this->pageProxies as $pid => $p) {
        $path = $p['path'];
        $defaults = [
          '_controller' => '\Drupal\page_proxy\Controller\PageProxy::serve',
          '_title' => $p['title'],
          'uri' => $p['uri'],
          'path' => $path,
          'cookie_prefix' => $p['cookie_prefix'],
          'cookie_allowlist' => $p['cookies'],
          'headers_request' => $p['headers_request'],
          'headers_response' => $p['headers_response'],
        ];
        $requirements = [];
        if (!empty($p['permission'])) {
          $requirements['_permission'] = $p['permission'];
        }
        else {
          $requirements['_access'] = TRUE;
        }

        $this->routes['page_proxy.' . $pid] = new Route(
          $path,
          $defaults,
          $requirements,
        );
        for ($i = 0; $i < $p['max_depth']; $i++) {
          $path .= "/{p$i}";
          $this->routes['page_proxy.' . $pid . '_d' . $i] = new Route(
            $path,
            $defaults,
            $requirements,
          );
        }
      }
    }
    return $this->routes;
  }

}
