<?php

namespace Drupal\page_refresh_webhook\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Page Refresh WebHook settings for this site.
 */
class PageRefreshWebhookConfigForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);

    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->keyRepository = $container->get('key.repository');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'page_refresh_webhook_config_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['page_refresh_webhook.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('page_refresh_webhook.settings');

    $form['endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('Webhook Endpoint'),
      '#description' => $this->t('Enter the URL of the endpoint to send the POST request.'),
      '#default_value' => $config->get('endpoint'),
      '#required' => TRUE,
    ];

    $key_options = [];
    foreach ($this->keyRepository->getKeys() as $key) {
      $key_options[$key->id()] = $key->label();
    }

    $form['api_key'] = [
      '#type' => 'select',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Select the key to use for authenticating webhook requests. Leave empty for unauthenticated requests.'),
      '#options' => $key_options,
      '#empty_option' => $this->t('- None -'),
      '#default_value' => $config->get('api_key'),
    ];

    $content_types = $this->entityTypeManager->getStorage('node_type')
      ->loadMultiple();
    $form['content_types'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Content Types and Depth'),
      '#tree' => TRUE,
    ];

    foreach ($content_types as $content_type) {
      $type_id = $content_type->id();
      $form['content_types'][$type_id] = [
        '#type' => 'fieldset',
        '#title' => $content_type->label(),
      ];
      $form['content_types'][$type_id]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable webhook for this content type'),
        '#default_value' => $config->get("content_types.$type_id.enabled"),
      ];
      $form['content_types'][$type_id]['depth'] = [
        '#type' => 'select',
        '#title' => $this->t('Default crawl depth'),
        '#options' => [
          1 => $this->t('1 - Only the specified URL'),
          2 => $this->t('2 - Specified URL and linked elements'),
        ],
        '#default_value' => $config->get("content_types.$type_id.depth") ?: 1,
        '#states' => [
          'visible' => [
            ':input[name="content_types[' . $type_id . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('page_refresh_webhook.settings');
    $config->set('endpoint', $form_state->getValue('endpoint'));
    $config->set('api_key', $form_state->getValue('api_key'));

    // Clear existing content type settings.
    $config->clear('content_types');

    $content_types = $form_state->getValue('content_types');
    foreach ($content_types as $type_id => $settings) {
      if ($settings['enabled']) {
        $config->set("content_types.$type_id", [
          'enabled' => $settings['enabled'],
          'depth' => $settings['depth'],
        ]);
      }
    }

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
