<?php

namespace Drupal\pagegeofence\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\pagegeofence\PageGeofenceManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for PageGeofence debug functionality.
 */
class PageGeofenceDebugController extends ControllerBase {

  /**
   * The page geofence manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceManager
   */
  protected $pageGeofenceManager;

  /**
   * Constructs a PageGeofenceDebugController object.
   *
   * @param \Drupal\pagegeofence\PageGeofenceManager $page_geofence_manager
   *   The page geofence manager.
   */
  public function __construct(PageGeofenceManager $page_geofence_manager) {
    $this->pageGeofenceManager = $page_geofence_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('pagegeofence.manager')
    );
  }

  /**
   * Debug country detection information.
   *
   * @return array
   *   A render array containing debug information.
   */
  public function debugInfo() {
    $debug_info = $this->pageGeofenceManager->getCountryDetectionDebugInfo();
    
    $build = [];
    
    $build['intro'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('This page shows debug information for country detection configuration and current request headers.') . '</p>',
    ];
    
    // Current country detection.
    $build['information'] = [
      '#type' => 'details',
      '#title' => $this->t('Information'),
      '#open' => TRUE,
    ];

    if (empty($debug_info['configured_header'])) {
      $build['configured_header']['warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . 
          $this->t('No country headers are configured! <a href="@settings_url">Configure headers</a> to enable country detection.', [
            '@settings_url' => '/admin/config/system/pagegeofence/settings',
          ]) . 
          '</div>',
      ];
    } else {
      $build['information']['list'] = [
        '#type' => 'markup',
        '#markup' => '<p><strong>' . $this->t('Configured Header:') . '</strong> ' . $debug_info['configured_header'] . '</p>',
      ];
			$build['information']['detected'] = [
				'#type' => 'markup',
				'#markup' => '<p><strong>' . $this->t('Detected Country:') . '</strong> ' . $debug_info['detected_country'] . '</p>',
			];
    }
    
    $build['actions'] = [
      '#type' => 'markup',
      '#markup' => '<p><a href="/admin/config/system/pagegeofence/settings" class="button">' . $this->t('Configure Headers') . '</a> ' .
        '<a href="/admin/config/system/pagegeofence" class="button">' . $this->t('Back to Rules') . '</a></p>',
    ];
    
    return $build;
  }

}