<?php

namespace Drupal\pagegeofence\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Path\PathMatcherInterface;

/**
 * Defines the Page Geofence Rule entity.
 *
 * @ConfigEntityType(
 *   id = "pagegeofence_rule",
 *   label = @Translation("Page Geofence Rule"),
 *   label_collection = @Translation("Page Geofence Rules"),
 *   label_singular = @Translation("page geofence rule"),
 *   label_plural = @Translation("page geofence rules"),
 *   label_count = @PluralTranslation(
 *     singular = "@count page geofence rule",
 *     plural = "@count page geofence rules",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\pagegeofence\PageGeofenceRuleListBuilder",
 *     "form" = {
 *       "add" = "Drupal\pagegeofence\Form\PageGeofenceRuleForm",
 *       "edit" = "Drupal\pagegeofence\Form\PageGeofenceRuleForm",
 *       "delete" = "Drupal\pagegeofence\Form\PageGeofenceRuleDeleteForm"
 *     }
 *   },
 *   config_prefix = "rule",
 *   admin_permission = "administer pagegeofence",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "status",
 *     "reason_for_restriction",
 *     "page_url",
 *     "restriction_type",
 *     "impacted_countries",
 *     "weight",
 *     "redirect_url"
 *   },
 *   links = {
 *     "add-form" = "/admin/config/system/pagegeofence/add",
 *     "edit-form" = "/admin/config/system/pagegeofence/{pagegeofence_rule}/edit",
 *     "delete-form" = "/admin/config/system/pagegeofence/{pagegeofence_rule}/delete",
 *     "collection" = "/admin/config/system/pagegeofence"
 *   }
 * )
 */
class PageGeofenceRule extends ConfigEntityBase {

  /**
   * The Page Geofence Rule ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The Page Geofence Rule label.
   *
   * @var string
   */
  protected $label;

  /**
   * The status of the rule.
   *
   * @var bool
   */
  protected $status = TRUE;

  /**
   * The reason for restriction.
   *
   * @var string
   */
  protected $reason_for_restriction;

  /**
   * The page URL pattern.
   *
   * @var string
   */
  protected $page_url;

  /**
   * The restriction type.
   *
   * @var string
   */
  protected $restriction_type;

  /**
   * The impacted countries.
   *
   * @var array
   */
  protected $impacted_countries = [];

  /**
   * The weight of the rule for ordering.
   *
   * @var int
   */
  protected $weight = 0;

  /**
   * The redirect URL for this rule.
   *
   * @var string
   */
  protected $redirect_url;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * Gets the reason for restriction.
   *
   * @return string
   *   The reason for restriction.
   */
  public function getReasonForRestriction() {
    return $this->reason_for_restriction;
  }

  /**
   * Sets the reason for restriction.
   *
   * @param string $reason
   *   The reason for restriction.
   *
   * @return $this
   */
  public function setReasonForRestriction($reason) {
    $this->reason_for_restriction = $reason;
    return $this;
  }

  /**
   * Gets the page URL.
   *
   * @return string
   *   The page URL.
   */
  public function getPageUrl() {
    return $this->page_url;
  }

  /**
   * Sets the page URL.
   *
   * @param string $page_url
   *   The page URL.
   *
   * @return $this
   */
  public function setPageUrl($page_url) {
    $this->page_url = $page_url;
    return $this;
  }

  /**
   * Gets the restriction type.
   *
   * @return string
   *   The restriction type.
   */
  public function getRestrictionType() {
    return $this->restriction_type;
  }

  /**
   * Sets the restriction type.
   *
   * @param string $restriction_type
   *   The restriction type.
   *
   * @return $this
   */
  public function setRestrictionType($restriction_type) {
    $this->restriction_type = $restriction_type;
    return $this;
  }

  /**
   * Gets the impacted countries.
   *
   * @return array
   *   The impacted countries.
   */
  public function getImpactedCountries() {
    return $this->impacted_countries ?? [];
  }

  /**
   * Sets the impacted countries.
   *
   * @param array $countries
   *   The impacted countries.
   *
   * @return $this
   */
  public function setImpactedCountries(array $countries) {
    $this->impacted_countries = array_values(array_filter($countries));
    return $this;
  }

  /**
   * Gets the weight.
   *
   * @return int
   *   The weight.
   */
  public function getWeight() {
    return $this->weight ?? 0;
  }

  /**
   * Sets the weight.
   *
   * @param int $weight
   *   The weight.
   *
   * @return $this
   */
  public function setWeight($weight) {
    $this->weight = (int) $weight;
    return $this;
  }

  /**
   * Gets the redirect URL.
   *
   * @return string
   *   The redirect URL.
   */
  public function getRedirectUrl() {
    return $this->redirect_url;
  }

  /**
   * Sets the redirect URL.
   *
   * @param string $redirect_url
   *   The redirect URL.
   *
   * @return $this
   */
  public function setRedirectUrl($redirect_url) {
    $this->redirect_url = $redirect_url;
    return $this;
  }

  /**
   * Gets the path matcher service.
   *
   * @return \Drupal\Core\Path\PathMatcherInterface
   *   The path matcher service.
   */
  protected function getPathMatcher() {
    if (!$this->pathMatcher) {
      $this->pathMatcher = \Drupal::service('path.matcher');
    }
    return $this->pathMatcher;
  }

  /**
   * Check if this rule applies to the given path.
   *
   * @param string $current_path
   *   The current path to check.
   *
   * @return bool
   *   TRUE if the rule applies to the path, FALSE otherwise.
   */
  public function appliesTo($current_path) {
    $configured_urls = $this->getPageUrl();

    // If no URLs configured, rule applies to all pages.
    if (empty($configured_urls)) {
      return TRUE;
    }

    // Normalize current path (remove trailing slashes).
    $current_path = rtrim($current_path, '/');
    
    // Handle root path.
    if (empty($current_path)) {
      $current_path = '/';
    } elseif (!str_starts_with($current_path, '/')) {
      $current_path = '/' . $current_path;
    }

    // Parse multiple URLs (one per line).
    $url_patterns = array_filter(array_map('trim', explode("\n", $configured_urls)));

    foreach ($url_patterns as $pattern) {
      // Normalize pattern (remove trailing slashes).
      $pattern = rtrim($pattern, '/');
      
      // Handle root path pattern.
      if (empty($pattern)) {
        $pattern = '/';
      } elseif (!str_starts_with($pattern, '/')) {
        $pattern = '/' . $pattern;
      }

      // Use Drupal's path matcher for consistent path matching.
      if ($this->getPathMatcher()->matchPath($current_path, $pattern)) {
        return TRUE;
      }
    }

    return FALSE;
  }



  /**
   * Check if the given country should be restricted.
   *
   * @param string $country_code
   *   The country code to check.
   *
   * @return bool
   *   TRUE if the country should be restricted, FALSE otherwise.
   */
  public function isCountryRestricted($country_code) {
    $impacted_countries = $this->getImpactedCountries();
    switch ($this->getRestrictionType()) {
      case 'allow_except':
        // Allow all except selected countries.
        return in_array($country_code, $impacted_countries);

      case 'deny_except':
        // Deny all except selected countries.
        return !in_array($country_code, $impacted_countries);

      default:
        return FALSE;
    }
  }

}