<?php

namespace Drupal\pagegeofence\EventSubscriber;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\pagegeofence\PageGeofenceManager;
use Drupal\pagegeofence\PageGeofenceWhitelistManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for page geofencing.
 */
class PageGeofenceSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The page geofence manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceManager
   */
  protected $geofenceManager;

  /**
   * The whitelist manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceWhitelistManager
   */
  protected $whitelistManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs a PageGeofenceSubscriber object.
   *
   * @param \Drupal\pagegeofence\PageGeofenceManager $geofence_manager
   *   The page geofence manager.
   * @param \Drupal\pagegeofence\PageGeofenceWhitelistManager $whitelist_manager
   *   The whitelist manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(PageGeofenceManager $geofence_manager, PageGeofenceWhitelistManager $whitelist_manager, AccountInterface $current_user) {
    $this->geofenceManager = $geofence_manager;
    $this->whitelistManager = $whitelist_manager;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Run early in the request cycle, but after routing.
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 50];
    return $events;
  }

  /**
   * Handles kernel request events.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The kernel request event.
   */
  public function onKernelRequest(RequestEvent $event) {
    // Only check main requests.
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();
    $current_path = $request->getPathInfo();

    // Skip admin routes and AJAX requests.
    if (strpos($current_path, '/admin') === 0 || $request->isXmlHttpRequest()) {
      return;
    }

    // Skip if user has permission to bypass geofencing.
    if ($this->currentUser->hasPermission('administer pagegeofence') || 
        $this->currentUser->hasPermission('bypass geofencing restrictions')) {
      return;
    }

    // Skip if the URL is whitelisted.
    if ($this->whitelistManager->isWhitelisted($current_path)) {
      return;
    }

    $user_country = $this->geofenceManager->getUserCountry();
    
    if ($user_country) {
      $restriction = $this->geofenceManager->isAccessRestricted($current_path, $user_country);        
			if ($restriction) {
				// Check if a redirect URL is configured.
				$redirect_url = $restriction['redirect_url'];
				
				if (!empty($redirect_url)) {
					// Handle internal vs external URLs.
					if (strpos($redirect_url, '/') === 0) {
						// Internal URL - convert to absolute URL.
						$request = $event->getRequest();
						$base_url = $request->getSchemeAndHttpHost();
						$redirect_url = $base_url . $redirect_url;
					}
					
					// Log the restriction with redirect action.
					$this->geofenceManager->logActivity('Access restricted for country @country on path @path by rule @rule - redirected to @redirect_url', [
							'@country' => $user_country,
							'@path' => $current_path,
							'@rule' => $restriction['rule']->label(),
							'@redirect_url' => $redirect_url,
					]);
					
					// Create a redirect response.
					$response = new Response('', 302, ['Location' => $redirect_url]);
					$event->setResponse($response);
				} else {
					// Log the restriction with 403 action.
					$this->geofenceManager->logActivity('Access restricted for country @country on path @path by rule @rule - showed 403 page', [
							'@country' => $user_country,
							'@path' => $current_path,
							'@rule' => $restriction['rule']->label(),
					]);
					
					// Create a 403 Forbidden response with the restriction reason.
					$content = $this->buildRestrictionPage($restriction['reason'], $user_country, $restriction['rule']);
					
					$response = new Response($content, 403);
					$event->setResponse($response);
				}
			}
    }
  }

  /**
   * Build the content for the restriction page.
   *
   * @param string $reason
   *   The reason for the restriction.
   * @param string $country
   *   The user's country code.
   * @param \Drupal\pagegeofence\Entity\PageGeofenceRule $rule
   *   The rule that triggered the restriction.
   *
   * @return string
   *   The HTML content for the restriction page.
   */
  protected function buildRestrictionPage($reason, $country, $rule = NULL) {
    $html = '<!DOCTYPE html>
<html>
<head>
  <title>Access Restricted</title>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <style>
    body {
      font-family: Arial, sans-serif;
      margin: 0;
      padding: 20px;
      background-color: #f5f5f5;
    }
    .container {
      max-width: 600px;
      margin: 50px auto;
      background: white;
      padding: 30px;
      border-radius: 8px;
      box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
    h1 {
      color: #d32f2f;
      margin-bottom: 20px;
    }
    .reason {
      background-color: #fff3cd;
      border: 1px solid #ffeaa7;
      border-radius: 4px;
      padding: 15px;
      margin: 20px 0;
    }
    .country-info {
      color: #666;
      font-size: 14px;
      margin-top: 20px;
    }
  </style>
</head>
<body>
  <div class="container">
    <h1>' . $this->t('Access Restricted') . '</h1>
    <p>' . $this->t('Sorry, this content is not available in your region.') . '</p>';
    if (!empty($reason)) {
      $html .= '<div class="reason">
        <strong>' . $this->t('Reason:') . '</strong><br>
        ' . htmlspecialchars($reason) . '
      </div>';
    }
    
    $html .= '<div class="country-info">
      ' . $this->t('Detected location: @country', ['@country' => $country]) . '
    </div>
  </div>
</body>
</html>';

    return $html;
  }

}