<?php

namespace Drupal\pagegeofence\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Page Geofence global settings for this site.
 */
class PageGeofenceSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'pagegeofence_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['pagegeofence.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('pagegeofence.settings');

    $form['whitelist'] = [
      '#type' => 'details',
      '#title' => $this->t('Whitelist URLs'),
      '#open' => TRUE,
    ];

    $form['whitelist']['whitelist_urls'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Whitelist URL Patterns'),
      '#description' => $this->t('Enter URL patterns that should never be geo-restricted, one per line. These URLs will be completely exempt from all geofencing rules. Supports wildcards (*). Examples:<br>/user/login<br>/user/register<br>/admin/*<br>/api/public/*'),
      '#default_value' => $config->get('whitelist_urls') ?? '',
      '#placeholder' => '/user/login' . "\n" . '/user/register' . "\n" . '/admin/*' . "\n" . '/api/public/*',
      '#rows' => 8,
    ];

    $form['whitelist']['help'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning">' . 
        $this->t('<strong>Important:</strong> Whitelisted URLs cannot be added to individual geofence rules. They are globally exempt from all restrictions.') . 
        '</div>',
    ];

    $form['global_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Global Settings'),
      '#open' => TRUE,
    ];

    $form['global_settings']['enable_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable detailed logging'),
      '#description' => $this->t('Log all geofencing activity including successful blocks and configuration changes.'),
      '#default_value' => $config->get('enable_logging') ?? TRUE,
    ];

    $form['country_detection'] = [
      '#type' => 'details',
      '#title' => $this->t('Country Detection Headers'),
      '#open' => TRUE,
    ];

    $form['country_detection']['country_header'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Country Header'),
      '#description' => $this->t('HTTP header to check for country code detection. This header should contain a valid ISO 3166-1 alpha-2 country code.'),
      '#default_value' => $config->get('country_header') ?? '',
      '#required' => TRUE,
    ];

    $form['country_detection']['help'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info">' . 
        $this->t('<strong>Common Headers:</strong><br>• CF-IPCountry (Cloudflare)<br>• X-Country-Code (Custom header)<br>• CloudFront-Viewer-Country (AWS CloudFront)<br>• X-Forwarded-Country<br>• X-GeoIP-Country') . 
        '</div>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate whitelist URLs format if provided.
    $whitelist_urls = trim($form_state->getValue('whitelist_urls'));
    if (!empty($whitelist_urls)) {
      $urls = array_filter(array_map('trim', explode("\n", $whitelist_urls)));
      foreach ($urls as $line_number => $url) {
        if (!preg_match('/^\//', $url)) {
          $form_state->setErrorByName('whitelist_urls', 
            $this->t('Each URL must start with a forward slash (/). Error on line @line: @url', [
              '@line' => $line_number + 1,
              '@url' => $url,
            ]));
          break;
        }
        // Validate wildcard usage.
        if (strpos($url, '*') !== false) {
          if (!preg_match('/^\/[^*]*(\*|\/\*)?$/', $url)) {
            $form_state->setErrorByName('whitelist_urls', 
              $this->t('Invalid wildcard usage on line @line: @url. Wildcards (*) can only be used at the end of a path or as complete path segments.', [
                '@line' => $line_number + 1,
                '@url' => $url,
              ]));
            break;
          }
        }
      }
    }

    // Validate country header - must be provided and in valid format.
    $country_header = trim($form_state->getValue('country_header'));
    if (empty($country_header)) {
      $form_state->setErrorByName('country_header', 
        $this->t('A country header must be specified.'));
    } else {
      // Validate header format.
      if (!preg_match('/^[A-Za-z0-9\-_]+$/', $country_header)) {
        $form_state->setErrorByName('country_header', 
          $this->t('Invalid header format: @header. Headers should only contain letters, numbers, hyphens, and underscores.', [
            '@header' => $country_header,
          ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Process single country header.
    $country_header = trim($form_state->getValue('country_header'));
    
    $this->config('pagegeofence.settings')
      ->set('whitelist_urls', $form_state->getValue('whitelist_urls'))
      ->set('enable_logging', $form_state->getValue('enable_logging'))
      ->set('country_header', $country_header)
      ->save();

    parent::submitForm($form, $form_state);
  }

}