<?php

namespace Drupal\pagegeofence;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for managing page geofencing functionality.
 */
class PageGeofenceManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * Constructs a PageGeofenceManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, RequestStack $request_stack, LoggerChannelFactoryInterface $logger_factory) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
    $this->loggerFactory = $logger_factory;
  }

  /**
   * Check if the current request should be restricted based on geofencing.
   *
   * @param string $current_path
   *   The current path to check.
   * @param string $user_country
   *   The user's country code (ISO 3166-1 alpha-2).
   *
   * @return array|false
   *   Array with restriction details if access should be restricted, FALSE otherwise.
   *   Array contains: 'restricted' => TRUE, 'reason' => string, 'rule' => PageGeofenceRule, 'redirect_url' => string|null
   */
  public function isAccessRestricted($current_path, $user_country) {
    $rules = $this->getActiveRules();
    
    foreach ($rules as $rule) {
      if ($rule->appliesTo($current_path) && $rule->isCountryRestricted($user_country)) {
        return [
          'restricted' => TRUE,
          'reason' => $rule->getReasonForRestriction(),
          'rule' => $rule,
          'redirect_url' => $rule->getRedirectUrl(),
        ];
      }
    }

    return FALSE;
  }

  /**
   * Get all active geofence rules ordered by weight.
   *
   * @return \Drupal\pagegeofence\Entity\PageGeofenceRule[]
   *   Array of active page geofence rules.
   */
  public function getActiveRules() {
    $storage = $this->entityTypeManager->getStorage('pagegeofence_rule');
    
    $rules = $storage->loadByProperties(['status' => TRUE]);
    
    // Sort by weight (lowest first).
    uasort($rules, function ($a, $b) {
      return $a->getWeight() <=> $b->getWeight();
    });
    return $rules;
  }



  /**
   * Get the user's country code from the request.
   *
   * This method checks configured headers in order to determine the user's country.
   *
   * @return string
   *   The user's country code (ISO 3166-1 alpha-2).
   */
  public function getUserCountry() {
    $request = $this->requestStack->getCurrentRequest();
    $return = FALSE;
    // Get configured country header.
    $config = $this->configFactory->get('pagegeofence.settings');
    $country_header = $config->get('country_header') ?? '';
    // Check the configured header.
    $country_code = $request->headers->get($country_header);
    if ($country_code && strlen($country_code) === 2) {
      $return = strtoupper($country_code);
    }
    return $return;
  }

  /**
   * Log geofencing activity.
   *
   * @param string $message
   *   The message to log.
   * @param array $context
   *   Additional context for the log entry.
   */
  public function logActivity($message, array $context = []) {
    // Only log if logging is enabled in configuration.
    $config = $this->configFactory->get('pagegeofence.settings');
    if (!($config->get('enable_logging') ?? TRUE)) {
      return;
    }
    
    $this->loggerFactory->get('pagegeofence')->info($message, $context);
  }

  /**
   * Get debugging information about country detection.
   *
   * @return array
   *   Array containing country detection debug information.
   */
  public function getCountryDetectionDebugInfo() {
    $request = $this->requestStack->getCurrentRequest();
    $config = $this->configFactory->get('pagegeofence.settings');
    $country_header = $config->get('country_header') ?? '';
    
    $debug_info = [
      'configured_header' => $country_header,
      'detected_country' => $this->getUserCountry(),
      'header_value' => $request->headers->get($country_header),
    ];
    
    return $debug_info;
  }

  /**
   * Get the restriction reason from the first matching rule.
   *
   * @param string $current_path
   *   The current path to check.
   * @param string $user_country
   *   The user's country code.
   *
   * @return string
   *   The restriction reason, or empty string if no restriction applies.
   */
  public function getRestrictionReason($current_path, $user_country) {
    $restriction = $this->isAccessRestricted($current_path, $user_country);
    return $restriction ? $restriction['reason'] : '';
  }

}