<?php

namespace Drupal\pagegeofence;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\pagegeofence\Entity\PageGeofenceRule;

/**
 * Service for logging PageGeofence rule configuration changes.
 * 
 * This service respects the 'enable_logging' configuration setting in
 * pagegeofence.settings. When logging is disabled, no log entries will
 * be created.
 * 
 * To test:
 * 1. Go to /admin/config/system/pagegeofence/settings
 * 2. Check/uncheck "Enable detailed logging"
 * 3. Create/edit/delete pagegeofence rules
 * 4. Check /admin/reports/dblog for pagegeofence_rules entries
 */
class PageGeofenceRuleLogger {

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a PageGeofenceRuleLogger object.
   *
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, AccountInterface $current_user, ConfigFactoryInterface $config_factory) {
    $this->loggerFactory = $logger_factory;
    $this->currentUser = $current_user;
    $this->configFactory = $config_factory;
  }

  /**
   * Logs when a PageGeofence rule is created.
   *
   * @param \Drupal\pagegeofence\Entity\PageGeofenceRule $entity
   *   The PageGeofence rule entity.
   */
  public function logRuleCreated(PageGeofenceRule $entity) {
    $this->logActivity('PageGeofence rule created: "@label" (ID: @id)', [
      '@label' => $entity->label(),
      '@id' => $entity->id(),
    ], 'info', [
      'rule_id' => $entity->id(),
      'rule_label' => $entity->label(),
      'status' => $entity->status() ? 'enabled' : 'disabled',
      'page_url' => $entity->getPageUrl(),
      'restriction_type' => $entity->getRestrictionType(),
      'impacted_countries' => $entity->getImpactedCountries(),
      'weight' => $entity->getWeight(),
      'reason_for_restriction' => $entity->getReasonForRestriction(),
      'user_id' => $this->currentUser->id(),
      'username' => $this->currentUser->getAccountName(),
    ]);
  }

  /**
   * Logs when a PageGeofence rule is updated.
   *
   * @param \Drupal\pagegeofence\Entity\PageGeofenceRule $entity
   *   The PageGeofence rule entity.
   * @param \Drupal\pagegeofence\Entity\PageGeofenceRule $original
   *   The original PageGeofence rule entity.
   */
  public function logRuleUpdated(PageGeofenceRule $entity, PageGeofenceRule $original = NULL) {
    if (!$original) {
      return;
    }

    $changes = [];
    
    // Track specific field changes.
    $current_values = [
      'label' => $entity->label(),
      'status' => $entity->status(),
      'reason_for_restriction' => $entity->getReasonForRestriction(),
      'page_url' => $entity->getPageUrl(),
      'restriction_type' => $entity->getRestrictionType(),
      'impacted_countries' => $entity->getImpactedCountries(),
      'weight' => $entity->getWeight(),
    ];

    $original_values = [
      'label' => $original->label(),
      'status' => $original->status(),
      'reason_for_restriction' => $original->getReasonForRestriction(),
      'page_url' => $original->getPageUrl(),
      'restriction_type' => $original->getRestrictionType(),
      'impacted_countries' => $original->getImpactedCountries(),
      'weight' => $original->getWeight(),
    ];

    foreach ($current_values as $field => $current_value) {
      $original_value = $original_values[$field] ?? NULL;
      
      if ($field === 'impacted_countries') {
        // Handle array comparison for countries.
        $current_sorted = $current_value;
        $original_sorted = $original_value;
        sort($current_sorted);
        sort($original_sorted);
        if ($current_sorted !== $original_sorted) {
          $changes[$field] = [
            'from' => $original_value,
            'to' => $current_value,
          ];
        }
      } else {
        if ($current_value !== $original_value) {
          $changes[$field] = [
            'from' => $original_value,
            'to' => $current_value,
          ];
        }
      }
    }

    if (!empty($changes)) {
      $change_summary = [];
      foreach ($changes as $field => $change) {
        if ($field === 'status') {
          $from_status = $change['from'] ? 'enabled' : 'disabled';
          $to_status = $change['to'] ? 'enabled' : 'disabled';
          $change_summary[] = "status: {$from_status} → {$to_status}";
        } elseif ($field === 'impacted_countries') {
          $from_countries = implode(', ', $change['from']);
          $to_countries = implode(', ', $change['to']);
          $change_summary[] = "countries: [{$from_countries}] → [{$to_countries}]";
        } else {
          $change_summary[] = "{$field}: \"{$change['from']}\" → \"{$change['to']}\"";
        }
      }

      $this->logActivity('PageGeofence rule updated: "@label" (ID: @id) - Changes: @changes', [
        '@label' => $entity->label(),
        '@id' => $entity->id(),
        '@changes' => implode(', ', $change_summary),
      ], 'info', [
        'rule_id' => $entity->id(),
        'rule_label' => $entity->label(),
        'changes' => $changes,
        'user_id' => $this->currentUser->id(),
        'username' => $this->currentUser->getAccountName(),
      ]);
    }
  }

  /**
   * Logs when a PageGeofence rule is deleted.
   *
   * @param \Drupal\pagegeofence\Entity\PageGeofenceRule $entity
   *   The PageGeofence rule entity.
   */
  public function logRuleDeleted(PageGeofenceRule $entity) {
    $this->logActivity('PageGeofence rule deleted: "@label" (ID: @id)', [
      '@label' => $entity->label(),
      '@id' => $entity->id(),
    ], 'warning', [
      'rule_id' => $entity->id(),
      'rule_label' => $entity->label(),
      'deleted_rule_config' => [
        'status' => $entity->status() ? 'enabled' : 'disabled',
        'page_url' => $entity->getPageUrl(),
        'restriction_type' => $entity->getRestrictionType(),
        'impacted_countries' => $entity->getImpactedCountries(),
        'weight' => $entity->getWeight(),
        'reason_for_restriction' => $entity->getReasonForRestriction(),
      ],
      'user_id' => $this->currentUser->id(),
      'username' => $this->currentUser->getAccountName(),
    ]);
  }

  /**
   * Check if logging is enabled in configuration.
   *
   * @return bool
   *   TRUE if logging is enabled, FALSE otherwise.
   */
  protected function isLoggingEnabled() {
    $config = $this->configFactory->get('pagegeofence.settings');
    return $config->get('enable_logging') ?? TRUE;
  }

  /**
   * Log activity with detailed context.
   *
   * @param string $message
   *   The message to log.
   * @param array $message_params
   *   Parameters for the message.
   * @param string $level
   *   The log level (info, warning, error).
   * @param array $context
   *   Additional context for the log entry.
   */
  public function logActivity($message, array $message_params = [], $level = 'info', array $context = []) {
    // Only log if logging is enabled in configuration.
    if (!$this->isLoggingEnabled()) {
      return;
    }

    $logger = $this->loggerFactory->get('pagegeofence_rules');
    
    // Add timestamp and user info to context.
    $context['timestamp'] = time();
    $context['date'] = date('Y-m-d H:i:s');
    
    switch ($level) {
      case 'warning':
        $logger->warning($message, $message_params + $context);
        break;
      case 'error':
        $logger->error($message, $message_params + $context);
        break;
      default:
        $logger->info($message, $message_params + $context);
    }
  }

}