<?php

namespace Drupal\pagegeofence\Tests;

use Drupal\Tests\BrowserTestBase;

/**
 * Test redirect functionality for Page Geofence rules.
 *
 * @group pagegeofence
 */
class PageGeofenceRedirectTest extends BrowserTestBase {

  /**
   * Modules to install.
   *
   * @var array
   */
  protected static $modules = ['pagegeofence'];

  /**
   * The theme to install as the default for testing.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * Test user with admin permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create an admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer pagegeofence',
      'bypass geofencing restrictions',
    ]);
  }

  /**
   * Test creating a rule with external redirect URL.
   */
  public function testExternalRedirectUrlField() {
    $this->drupalLogin($this->adminUser);
    
    // Go to the add rule form.
    $this->drupalGet('/admin/config/system/pagegeofence/add');
    
    // Check that the redirect URL field is present.
    $this->assertSession()->fieldExists('redirect_url');
    
    // Fill out the form with external redirect URL.
    $edit = [
      'label' => 'Test External Redirect Rule',
      'id' => 'test_external_redirect_rule',
      'reason_for_restriction' => 'This is a test restriction for external redirect functionality. This field requires at least fifty characters.',
      'page_url' => '/test-page',
      'restriction_type' => 'allow_except',
      'impacted_countries[US]' => 'US',
      'redirect_url' => 'https://example.com/blocked',
      'weight' => 0,
    ];
    
    $this->drupalPostForm(NULL, $edit, 'Save');
    
    // Check that the rule was created successfully.
    $this->assertSession()->pageTextContains('Created the Test External Redirect Rule page geofence rule.');
    
    // Check that the redirect URL is shown in the list.
    $this->assertSession()->pageTextContains('Redirect to example.com');
  }

  /**
   * Test creating a rule with internal redirect URL.
   */
  public function testInternalRedirectUrlField() {
    $this->drupalLogin($this->adminUser);
    
    // Go to the add rule form.
    $this->drupalGet('/admin/config/system/pagegeofence/add');
    
    // Fill out the form with internal redirect URL.
    $edit = [
      'label' => 'Test Internal Redirect Rule',
      'id' => 'test_internal_redirect_rule',
      'reason_for_restriction' => 'This is a test restriction for internal redirect functionality. This field requires at least fifty characters.',
      'page_url' => '/restricted-section',
      'restriction_type' => 'allow_except',
      'impacted_countries[US]' => 'US',
      'redirect_url' => '/access-denied',
      'weight' => 0,
    ];
    
    $this->drupalPostForm(NULL, $edit, 'Save');
    
    // Check that the rule was created successfully.
    $this->assertSession()->pageTextContains('Created the Test Internal Redirect Rule page geofence rule.');
    
    // Check that the redirect URL is shown in the list.
    $this->assertSession()->pageTextContains('Redirect to /access-denied');
  }

  /**
   * Test creating a rule without redirect URL (default 403 behavior).
   */
  public function testDefault403Behavior() {
    $this->drupalLogin($this->adminUser);
    
    // Go to the add rule form.
    $this->drupalGet('/admin/config/system/pagegeofence/add');
    
    // Fill out the form without redirect URL.
    $edit = [
      'label' => 'Test 403 Rule',
      'id' => 'test_403_rule',
      'reason_for_restriction' => 'This is a test restriction for default 403 behavior. This field requires at least fifty characters.',
      'page_url' => '/test-403-page',
      'restriction_type' => 'allow_except',
      'impacted_countries[US]' => 'US',
      'redirect_url' => '', // Empty redirect URL.
      'weight' => 0,
    ];
    
    $this->drupalPostForm(NULL, $edit, 'Save');
    
    // Check that the rule was created successfully.
    $this->assertSession()->pageTextContains('Created the Test 403 Rule page geofence rule.');
    
    // Check that the default behavior is shown in the list.
    $this->assertSession()->pageTextContains('Show 403 page');
  }

  /**
   * Test redirect URL validation for invalid formats.
   */
  public function testInvalidRedirectUrlValidation() {
    $this->drupalLogin($this->adminUser);
    
    // Go to the add rule form.
    $this->drupalGet('/admin/config/system/pagegeofence/add');
    
    // Test invalid URL.
    $edit = [
      'label' => 'Test Invalid URL',
      'id' => 'test_invalid_url',
      'reason_for_restriction' => 'This is a test restriction with invalid URL. This field requires at least fifty characters.',
      'page_url' => '/test-invalid',
      'restriction_type' => 'allow_except',
      'impacted_countries[US]' => 'US',
      'redirect_url' => 'not-a-valid-url',
      'weight' => 0,
    ];
    
    $this->drupalPostForm(NULL, $edit, 'Save');
    
    // Check for validation error.
    $this->assertSession()->pageTextContains('Please enter a valid URL. For external URLs use http:// or https://, for internal URLs start with /.');
  }

  /**
   * Test redirect URL validation to prevent loops.
   */
  public function testRedirectUrlLoopValidation() {
    $this->drupalLogin($this->adminUser);
    
    // Go to the add rule form.
    $this->drupalGet('/admin/config/system/pagegeofence/add');
    
    // Test redirect URL that matches page URL pattern.
    $edit = [
      'label' => 'Test Loop Prevention',
      'id' => 'test_loop_prevention',
      'reason_for_restriction' => 'This is a test restriction to prevent redirect loops. This field requires at least fifty characters.',
      'page_url' => '/restricted-section',
      'restriction_type' => 'allow_except',
      'impacted_countries[US]' => 'US',
      'redirect_url' => '/restricted-section',
      'weight' => 0,
    ];
    
    $this->drupalPostForm(NULL, $edit, 'Save');
    
    // Check for validation error.
    $this->assertSession()->pageTextContains('Redirect URL "/restricted-section" cannot be the same as or match the restricted URL pattern "/restricted-section". This would create a redirect loop.');
  }

}