<?php

namespace Drupal\pagegeofence;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;

/**
 * Provides a listing of Page Geofence Rule entities.
 */
class PageGeofenceRuleListBuilder extends ConfigEntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('Name');
    $header['status'] = $this->t('Status');
    $header['page_url'] = $this->t('URL Patterns');
    $header['restriction_type'] = $this->t('Restriction Type');
    $header['countries'] = $this->t('Countries');
    $header['redirect_url'] = $this->t('Action');
    $header['weight'] = $this->t('Weight');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\pagegeofence\Entity\PageGeofenceRule $entity */
    $row['label'] = $entity->label();
    $row['status'] = $entity->status() ? $this->t('Enabled') : $this->t('Disabled');
    
    // Handle multiple URLs display.
    $page_urls = $entity->getPageUrl();
    if (empty($page_urls)) {
      $row['page_url'] = $this->t('All pages');
    } else {
      $urls = array_filter(array_map('trim', explode("\n", $page_urls)));
      if (count($urls) == 1) {
        $row['page_url'] = $urls[0];
      } else {
        $row['page_url'] = count($urls) . ' ' . $this->formatPlural(count($urls), 'pattern', 'patterns');
      }
    }
    
    $restriction_options = [
      'allow_except' => $this->t('Allow all except selected'),
      'deny_except' => $this->t('Deny all except selected'),
    ];
    $row['restriction_type'] = $restriction_options[$entity->getRestrictionType()] ?? $entity->getRestrictionType();
    
    $countries = $entity->getImpactedCountries();
    $row['countries'] = count($countries) . ' ' . $this->formatPlural(count($countries), 'country', 'countries');
    
    // Display redirect action type.
    $redirect_url = $entity->getRedirectUrl();
    if (!empty($redirect_url)) {
      if (strpos($redirect_url, '/') === 0) {
        // Internal URL.
        $display_url = strlen($redirect_url) > 30 ? substr($redirect_url, 0, 27) . '...' : $redirect_url;
        $row['redirect_url'] = $this->t('Redirect to @url', ['@url' => $display_url]);
      } else {
        // External URL.
        $host = parse_url($redirect_url, PHP_URL_HOST);
        $row['redirect_url'] = $this->t('Redirect to @url', ['@url' => $host ?: $redirect_url]);
      }
    } else {
      $row['redirect_url'] = $this->t('Show 403 page');
    }
    
    $row['weight'] = $entity->getWeight();
    
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);
    
    // Add a toggle status operation.
    if ($entity->status()) {
      $operations['disable'] = [
        'title' => $this->t('Disable'),
        'weight' => 40,
        'url' => Url::fromRoute('entity.pagegeofence_rule.disable', [
          'pagegeofence_rule' => $entity->id(),
        ]),
      ];
    }
    else {
      $operations['enable'] = [
        'title' => $this->t('Enable'),
        'weight' => 40,
        'url' => Url::fromRoute('entity.pagegeofence_rule.enable', [
          'pagegeofence_rule' => $entity->id(),
        ]),
      ];
    }

    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();
    
    $build['table']['#empty'] = $this->t('There are no page geofence rules yet. <a href=":add-url">Add one</a>.', [
      ':add-url' => Url::fromRoute('entity.pagegeofence_rule.add_form')->toString(),
    ]);

    // Add some help text.
    $build['help'] = [
      '#type' => 'details',
      '#title' => $this->t('About Page Geofence Rules'),
      '#open' => FALSE,
      '#weight' => -10,
      'content' => [
        '#markup' => $this->t('<p>Page Geofence Rules allow you to restrict access to specific pages or sections of your site based on the visitor\'s geographic location. Rules are processed in order of weight (lowest first).</p><p>Each rule can:</p><ul><li><strong>Target specific pages</strong> with exact matches or wildcard patterns</li><li><strong>Use wildcards (*)</strong> to match multiple pages (e.g., /about/* for all about pages)</li><li><strong>Allow or deny</strong> access based on country selection</li><li><strong>Redirect users</strong> to an internal/external URL or show a 403 page</li><li><strong>Be enabled or disabled</strong> without deletion</li></ul>'),
      ],
    ];

    return $build;
  }

}