<?php

namespace Drupal\pagegeofence;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Path\PathMatcherInterface;

/**
 * Service for managing whitelisted URLs.
 */
class PageGeofenceWhitelistManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * Constructs a PageGeofenceWhitelistManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Path\PathMatcherInterface $path_matcher
   *   The path matcher service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, PathMatcherInterface $path_matcher) {
    $this->configFactory = $config_factory;
    $this->pathMatcher = $path_matcher;
  }

  /**
   * Check if a URL is whitelisted.
   *
   * @param string $path
   *   The path to check.
   *
   * @return bool
   *   TRUE if the path is whitelisted, FALSE otherwise.
   */
  public function isWhitelisted($path) {
    $whitelist_urls = $this->getWhitelistUrls();
    
    if (empty($whitelist_urls)) {
      return FALSE;
    }

    // Normalize path (remove trailing slashes and ensure leading slash).
    $path = rtrim($path, '/');
    if (empty($path)) {
      $path = '/';
    } elseif (!str_starts_with($path, '/')) {
      $path = '/' . $path;
    }

    foreach ($whitelist_urls as $pattern) {
      // Normalize pattern (remove trailing slashes and ensure leading slash).
      $pattern = rtrim($pattern, '/');
      if (empty($pattern)) {
        $pattern = '/';
      } elseif (!str_starts_with($pattern, '/')) {
        $pattern = '/' . $pattern;
      }

      // Use Drupal's path matcher for consistent path matching.
      if ($this->pathMatcher->matchPath($path, $pattern)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Get all whitelist URL patterns.
   *
   * @return array
   *   Array of whitelist URL patterns.
   */
  public function getWhitelistUrls() {
    $config = $this->configFactory->get('pagegeofence.settings');
    $whitelist_urls = $config->get('whitelist_urls');
    
    if (empty($whitelist_urls)) {
      return [];
    }

    return array_filter(array_map('trim', explode("\n", $whitelist_urls)));
  }



  /**
   * Check if any of the given URL patterns conflict with whitelist.
   *
   * @param string $url_patterns
   *   URL patterns to check (newline separated).
   *
   * @return array
   *   Array of conflicting patterns, empty if no conflicts.
   */
  public function checkForConflicts($url_patterns) {
    if (empty($url_patterns)) {
      return [];
    }

    $whitelist_urls = $this->getWhitelistUrls();
    if (empty($whitelist_urls)) {
      return [];
    }

    $patterns = array_filter(array_map('trim', explode("\n", $url_patterns)));
    $conflicts = [];

    foreach ($patterns as $pattern) {
      foreach ($whitelist_urls as $whitelist_pattern) {
        if ($this->patternsOverlap($pattern, $whitelist_pattern)) {
          $conflicts[] = [
            'pattern' => $pattern,
            'whitelist_pattern' => $whitelist_pattern,
          ];
        }
      }
    }

    return $conflicts;
  }

  /**
   * Check if two URL patterns overlap.
   *
   * @param string $pattern1
   *   First pattern.
   * @param string $pattern2
   *   Second pattern.
   *
   * @return bool
   *   TRUE if patterns overlap, FALSE otherwise.
   */
  protected function patternsOverlap($pattern1, $pattern2) {
    // Normalize patterns (remove trailing slashes and ensure leading slash).
    $pattern1 = rtrim($pattern1, '/');
    $pattern2 = rtrim($pattern2, '/');
    
    if (empty($pattern1)) {
      $pattern1 = '/';
    } elseif (!str_starts_with($pattern1, '/')) {
      $pattern1 = '/' . $pattern1;
    }
    
    if (empty($pattern2)) {
      $pattern2 = '/';
    } elseif (!str_starts_with($pattern2, '/')) {
      $pattern2 = '/' . $pattern2;
    }

    // Use PathMatcher to check if patterns overlap.
    // Check if pattern1 matches pattern2 or vice versa.
    return $this->pathMatcher->matchPath($pattern1, $pattern2) || 
           $this->pathMatcher->matchPath($pattern2, $pattern1);
  }

}