# Page Geofence Module

## Overview
The Page Geofence module provides geofencing functionality to restrict page access based on geographical locations. It allows administrators to create multiple geofencing rules to restrict access to specific pages or sections of a Drupal site based on the visitor's country.

## Features
- **Multiple Rules**: Create and manage multiple geofencing rules
- **Rule Management**: Enable/disable individual rules without deletion
- **Flexible Response Types**:
  - **Show Content**: Display custom restriction page with configurable content
  - **Redirect**: Send users to internal or external URLs
- **Customizable Restriction Content**:
  - Custom page titles, summaries, and formatted restriction reasons
  - Country-specific information display
  - Full rich text formatting support
  - Global defaults with per-rule overrides
- **Country Name Display**: Automatically converts country codes to full country names using Drupal's CountryManager
- **URL Pattern Matching**: Support for exact matches and wildcard patterns
- **Restriction Types**: 
  - Allow all except selected countries
  - Deny all except selected countries
- **Country Selection**: Multi-select dropdown with ISO 3166-1 alpha-2 country codes
- **Reason Documentation**: Document legal/business reasons for restrictions with minimum character requirements
- **Rule Ordering**: Weight-based rule processing (lowest weight processed first)
- **Validation**: Built-in validation for minimum character requirements and country selection
- **Translation Support**: All content fields are translatable via Configuration Translation module

## Installation
1. Download and install the module using Composer.
2. Enable the module via Drush: `drush en pagegeofence`
3. Or enable via the admin interface at `/admin/modules`

## Configuration

### Global Settings
1. Navigate to `/admin/config/system/pagegeofence/settings`
2. Configure whitelist URLs that should never be geo-restricted
3. Set global preferences for logging and default messages
4. **Configure Country Detection Headers** (mandatory - see below)
5. **Global Restriction Content Defaults** (optional):
   - **Restriction Page Title**: Default title for restriction pages (can be overridden per rule)
   - **Restriction Page Summary**: Default summary text (can be overridden per rule)
   - **Reason for Restriction**: Default reason with rich text formatting (can be overridden per rule)
   - **Show Country Information**: Global toggle for displaying country information
   - **Country Information Text**: Default country-specific text (can be overridden per rule)

### Geofence Rules
1. Navigate to `/admin/config/system/pagegeofence`
2. Click "Add Page Geofence Rule" to create a new rule
3. Configure each rule with:
   - **Rule Name**: Descriptive name for the rule
   - **Status**: Enable/disable the rule
   - **Page URL Patterns**: Specify URL patterns to restrict, one per line (leave empty for all pages, supports wildcards)
   - **Rule Type**: Select how to handle restricted access:
     - **Redirect Page**: Redirect users to a different URL
     - **Show Content**: Display custom restriction content
   - **Restriction Content** (when "Show Content" is selected):
     - **Restriction Page Title**: Title displayed on restriction page (falls back to global setting if empty)
     - **Restriction Page Summary**: Brief summary text (falls back to global setting if empty)
     - **Reason for Restriction**: Legal or business reason with rich text formatting (minimum 50 characters, falls back to global setting if empty)
     - **Show Country Information**: Toggle to display country-specific information
     - **Country Information Text**: Additional country information text (falls back to global setting if empty)
   - **Redirect URL** (when "Redirect Page" is selected): Internal path (e.g., `/access-denied`) or external URL
   - **Restriction Type**: Choose allow-except or deny-except logic
   - **Impacted Countries**: Select countries to be affected by the restriction
   - **Weight**: Control rule processing order (lower numbers processed first)

## Whitelist URLs
URLs added to the whitelist are globally exempt from all geofencing restrictions:
- **Complete exemption**: Whitelisted URLs bypass all geofence rules
- **Wildcard support**: Use `*` for pattern matching (e.g., `/admin/*`, `/user/*`)
- **Conflict prevention**: Rules cannot target URLs that are whitelisted
- **Common examples**: Login pages, registration, admin areas, public APIs

### Processing Order:
1. **Permission check**: Users with bypass permissions skip all restrictions
2. **Whitelist check**: Whitelisted URLs are allowed regardless of rules
3. **Geofence rules**: Remaining URLs are checked against active rules (by weight)

## Permissions
- **Administer Page Geofence**: Allows users to configure geofencing settings
- Users with this permission can bypass geofencing restrictions

## Technical Details

### Country Detection Configuration
**IMPORTANT**: Before creating any geofence rules, you must configure country detection headers.

#### Header Configuration
1. Navigate to `/admin/config/system/pagegeofence/settings`
2. Under "Country Detection Headers", configure the HTTP headers that contain country information
3. Headers are checked in order until a valid 2-letter country code is found
4. **At least one header must be configured** for the module to work

#### Default Headers
The module comes pre-configured with common headers:
- `CF-IPCountry` (Cloudflare)
- `X-Country-Code` (Custom header)
- `CloudFront-Viewer-Country` (AWS CloudFront)

#### Common CDN/Proxy Headers
- **Cloudflare**: `CF-IPCountry` (automatically set)
- **AWS CloudFront**: `CloudFront-Viewer-Country` (requires configuration)
- **Fastly**: `X-Country-Code` (requires VCL configuration)
- **Custom**: Set any header name that your infrastructure provides

### URL Matching
- **Multiple URL Support**: Enter multiple URL patterns, one per line
- **Wildcard Support**: Use `*` for pattern matching (e.g., `/admin/*`, `/career/*/jobs`)
- **Exact Match**: Matches only the specified path exactly (when no wildcards)
- **Wildcard Patterns**: Use `*` for flexible pattern matching

### URL Pattern Examples:
```
/admin/*                    # Matches all admin pages
/career/award/secret-page   # Matches exact page only
/restricted-section         # Matches exact path only
/api/*/private              # Matches any private API endpoint
```

### Restriction Logic
- **Allow all except selected**: Blocks access only for selected countries
- **Deny all except selected**: Allows access only for selected countries

## Development and Testing
**Important**: Ensure country detection headers are configured before testing geofence rules.

## Logging
The module logs geofencing activities to the 'pagegeofence' log channel for monitoring and debugging purposes.

## Requirements
- Drupal 11.x
- Core modules: field, system, user

## Rule Processing
Rules are processed in order of weight (lowest first). The first matching rule that applies to the current path and restricts the user's country will be used. This allows for:
- General rules with higher weights as fallbacks
- Specific rules with lower weights for exceptions
- Hierarchical rule organization

## API Usage
You can programmatically check restrictions using the service:

```php
$geofence_manager = \Drupal::service('pagegeofence.manager');
$restriction = $geofence_manager->isAccessRestricted('/some/path', 'DE');

if ($restriction) {
  // Access is restricted
  $reason = $restriction['reason'];
  $rule = $restriction['rule'];
  $page_title = $restriction['page_title'];
  $page_summary = $restriction['page_summary'];
  $country_info_flag = $restriction['country_information_flag'];
  $country_info_text = $restriction['country_information_text'];
}

// Get all active rules
$active_rules = $geofence_manager->getActiveRules();

// Convert country code to full country name
$country_name = $geofence_manager->getCountryName('US'); // Returns "United States"
```

## Entity Management
You can also work with the rules programmatically:

```php
// Load a specific rule
$rule = \Drupal::entityTypeManager()
  ->getStorage('pagegeofence_rule')
  ->load('my_rule_id');

// Create a new rule with custom content
$rule = \Drupal::entityTypeManager()
  ->getStorage('pagegeofence_rule')
  ->create([
    'id' => 'my_new_rule',
    'label' => 'My New Rule',
    'status' => TRUE,
    'restriction_page_title' => 'Access Restricted',
    'restriction_page_summary' => 'This content is not available in your region.',
    'reason_for_restriction' => [
      'value' => 'This content is restricted due to local regulations and licensing agreements.',
      'format' => 'basic_html',
    ],
    'country_information_flag' => TRUE,
    'country_information_text' => 'For more information, contact your regional office.',
    'page_url' => '/restricted-section/*',
    'restriction_type' => 'allow_except',
    'impacted_countries' => ['CN', 'RU'],
    'redirect_url' => '', // Empty for show content, or set URL for redirect
    'weight' => 10,
  ]);
$rule->save();
```

## Translation Support
The module supports multilingual sites through Drupal's Configuration Translation module:

1. Enable the **Configuration Translation** module (`config_translation`)
2. Navigate to `/admin/config/regional/config-translation`
3. Find **Page Geofence Rule** in the list
4. Click **Translate** to add translations for:
   - Rule label
   - Restriction page title
   - Restriction page summary
   - Reason for restriction
   - Country information text

Translated content will be displayed based on the current site language.

## Support
This module is designed for internal use and customization. Modify the country detection logic as needed for your specific infrastructure and requirements.