<?php

namespace Drupal\pagegeofence\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for displaying the geofence restriction page.
 */
class PageGeofenceRestrictionController extends ControllerBase {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a PageGeofenceRestrictionController object.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   */
  public function __construct(RequestStack $request_stack) {
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')
    );
  }

  /**
   * Gets the dynamic page title.
   *
   * @return string
   *   The page title from restriction data.
   */
  public function getTitle() {
    $request = $this->requestStack->getCurrentRequest();
    $restriction = $request->attributes->get('pagegeofence_restriction');
    return $restriction['page_title'] ?? $this->t('Access Restricted');
  }

  /**
   * Checks access to the restriction page.
   *
   * Only allow access if called via subrequest with restriction data.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access() {
    $request = $this->requestStack->getCurrentRequest();

    // Only allow access if restriction data is present (set by subrequest).
    $restriction = $request->attributes->get('pagegeofence_restriction');
    $country = $request->attributes->get('pagegeofence_country');

    if ($restriction && $country) {
      return AccessResult::allowed();
    }

    // Deny access if called directly without restriction data.
    return AccessResult::forbidden();
  }

  /**
   * Displays the restriction page.
   *
   * @return array
   *   A render array for the restriction page.
   */
  public function restrictionPage() {
    $request = $this->requestStack->getCurrentRequest();

    // Get restriction data from request attributes.
    $restriction = $request->attributes->get('pagegeofence_restriction');
    $country = $request->attributes->get('pagegeofence_country');

    // Return the themed restriction page.
    return [
      '#theme' => 'pagegeofence_restriction_page',
      '#page_title' => $restriction['page_title'],
      '#restriction_summary' => $restriction['restriction_summary'],
      '#reason' => $restriction['reason'],
      '#country' => $country,
      '#country_info_flag' => $restriction['country_info_flag'],
      '#country_info_text' => $restriction['country_info_text'],
      '#rule' => $restriction['rule'],
    ];
  }

}
