<?php

namespace Drupal\pagegeofence\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the Page Geofence Rule entity.
 *
 * @ConfigEntityType(
 *   id = "pagegeofence_rule",
 *   label = @Translation("Page Geofence Rule"),
 *   label_collection = @Translation("Page Geofence Rules"),
 *   label_singular = @Translation("page geofence rule"),
 *   label_plural = @Translation("page geofence rules"),
 *   label_count = @PluralTranslation(
 *     singular = "@count page geofence rule",
 *     plural = "@count page geofence rules",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\pagegeofence\PageGeofenceRuleListBuilder",
 *     "form" = {
 *       "add" = "Drupal\pagegeofence\Form\PageGeofenceRuleForm",
 *       "edit" = "Drupal\pagegeofence\Form\PageGeofenceRuleForm",
 *       "delete" = "Drupal\pagegeofence\Form\PageGeofenceRuleDeleteForm"
 *     }
 *   },
 *   config_prefix = "rule",
 *   admin_permission = "administer pagegeofence",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "status",
 *     "restriction_page_title",
 *     "restriction_page_summary",
 *     "reason_for_restriction",
 *     "page_url",
 *     "restriction_type",
 *     "impacted_countries",
 *     "country_information_flag",
 *     "country_information_text",
 *     "weight",
 *     "redirect_url"
 *   },
 *   links = {
 *     "add-form" = "/admin/config/system/pagegeofence/add",
 *     "edit-form" = "/admin/config/system/pagegeofence/{pagegeofence_rule}/edit",
 *     "delete-form" = "/admin/config/system/pagegeofence/{pagegeofence_rule}/delete",
 *     "collection" = "/admin/config/system/pagegeofence"
 *   }
 * )
 */
class PageGeofenceRule extends ConfigEntityBase {

  /**
   * The Page Geofence Rule ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The Page Geofence Rule label.
   *
   * @var string
   */
  protected $label;

  /**
   * The status of the rule.
   *
   * @var bool
   */
  protected $status = TRUE;

  /**
   * The restriction page title.
   *
   * @var string
   */
  protected $restriction_page_title;

  /**
   * The restriction page summary.
   *
   * @var string
   */
  protected $restriction_page_summary;

  /**
   * The reason for restriction.
   *
   * @var string
   */
  protected $reason_for_restriction;

  /**
   * The country information flag.
   *
   * @var bool
   */
  protected $country_information_flag;

  /**
   * The country information text.
   *
   * @var string
   */
  protected $country_information_text;

  /**
   * The page URL pattern.
   *
   * @var string
   */
  protected $page_url;

  /**
   * The restriction type.
   *
   * @var string
   */
  protected $restriction_type;

  /**
   * The impacted countries.
   *
   * @var array
   */
  protected $impacted_countries = [];

  /**
   * The weight of the rule for ordering.
   *
   * @var int
   */
  protected $weight = 0;

  /**
   * The redirect URL for this rule.
   *
   * @var string
   */
  protected $redirect_url;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * Gets the reason for restriction.
   *
   * @return array
   *   The reason for restriction with 'value' and 'format' keys.
   */
  public function getReasonForRestriction() {
    return $this->reason_for_restriction ?? ['value' => '', 'format' => ''];
  }

  /**
   * Gets the formatted reason for restriction.
   *
   * @return \Drupal\Component\Render\MarkupInterface|null
   *   The formatted reason for restriction markup, or NULL if empty.
   */
  public function getFormattedReasonForRestriction() {
    $reason = $this->getReasonForRestriction();
    if (!empty($reason['value'])) {
      return check_markup($reason['value'], $reason['format']);
    }
    return NULL;
  }

  /**
   * Sets the reason for restriction.
   *
   * @param array|string $reason
   *   The reason for restriction. Can be a string or array with 'value' and 'format'.
   *
   * @return $this
   */
  public function setReasonForRestriction($reason) {
    if (is_array($reason)) {
      $this->reason_for_restriction = $reason;
    }
    else {
      // Backwards compatibility for string values.
      $this->reason_for_restriction = [
        'value' => $reason,
        'format' => 'basic_html',
      ];
    }
    return $this;
  }

  /**
   * Gets the page URL.
   *
   * @return string
   *   The page URL.
   */
  public function getPageUrl() {
    return $this->page_url;
  }

  /**
   * Sets the page URL.
   *
   * @param string $page_url
   *   The page URL.
   *
   * @return $this
   */
  public function setPageUrl($page_url) {
    $this->page_url = $page_url;
    return $this;
  }

  /**
   * Gets the restriction type.
   *
   * @return string
   *   The restriction type.
   */
  public function getRestrictionType() {
    return $this->restriction_type;
  }

  /**
   * Sets the restriction type.
   *
   * @param string $restriction_type
   *   The restriction type.
   *
   * @return $this
   */
  public function setRestrictionType($restriction_type) {
    $this->restriction_type = $restriction_type;
    return $this;
  }

  /**
   * Gets the impacted countries.
   *
   * @return array
   *   The impacted countries.
   */
  public function getImpactedCountries() {
    return $this->impacted_countries ?? [];
  }

  /**
   * Sets the impacted countries.
   *
   * @param array $countries
   *   The impacted countries.
   *
   * @return $this
   */
  public function setImpactedCountries(array $countries) {
    $this->impacted_countries = array_values(array_filter($countries));
    return $this;
  }

  /**
   * Gets the weight.
   *
   * @return int
   *   The weight.
   */
  public function getWeight() {
    return $this->weight ?? 0;
  }

  /**
   * Sets the weight.
   *
   * @param int $weight
   *   The weight.
   *
   * @return $this
   */
  public function setWeight($weight) {
    $this->weight = (int) $weight;
    return $this;
  }

  /**
   * Gets the redirect URL.
   *
   * @return string
   *   The redirect URL.
   */
  public function getRedirectUrl() {
    return $this->redirect_url;
  }

  /**
   * Sets the redirect URL.
   *
   * @param string $redirect_url
   *   The redirect URL.
   *
   * @return $this
   */
  public function setRedirectUrl($redirect_url) {
    $this->redirect_url = $redirect_url;
    return $this;
  }

  /**
   * Gets the path matcher service.
   *
   * @return \Drupal\Core\Path\PathMatcherInterface
   *   The path matcher service.
   */
  protected function getPathMatcher() {
    if (!$this->pathMatcher) {
      $this->pathMatcher = \Drupal::service('path.matcher');
    }
    return $this->pathMatcher;
  }

  /**
   * Gets the restriction page title.
   *
   * @return string
   *   The restriction page title.
   */
  public function getRestrictionPageTitle() {
    return $this->restriction_page_title;
  }

  /**
   * Sets the restriction page title.
   *
   * @param string $title
   *   The restriction page title.
   *
   * @return $this
   */
  public function setRestrictionPageTitle($title) {
    $this->restriction_page_title = $title;
    return $this;
  }

  /**
   * Gets the restriction page summary.
   *
   * @return string
   *   The restriction page summary.
   */
  public function getRestrictionPageSummary() {
    return $this->restriction_page_summary;
  }

  /**
   * Sets the restriction page summary.
   *
   * @param string $summary
   *   The restriction page summary.
   *
   * @return $this
   */
  public function setRestrictionPageSummary($summary) {
    $this->restriction_page_summary = $summary;
    return $this;
  }

  /**
   * Gets the country information flag.
   *
   * @return bool
   *   The country information flag.
   */
  public function getCountryInformationFlag() {
    return $this->country_information_flag;
  }

  /**
   * Sets the country information flag.
   *
   * @param bool $flag
   *   The country information flag.
   *
   * @return $this
   */
  public function setCountryInformationFlag($flag) {
    $this->country_information_flag = (bool) $flag;
    return $this;
  }

  /**
   * Gets the country information text.
   *
   * @return string
   *   The country information text.
   */
  public function getCountryInformationText() {
    return $this->country_information_text;
  }

  /**
   * Sets the country information text.
   *
   * @param string $text
   *   The country information text.
   *
   * @return $this
   */
  public function setCountryInformationText($text) {
    $this->country_information_text = $text;
    return $this;
  }

  /**
   * Check if this rule applies to the given path.
   *
   * @param string $current_path
   *   The current path to check.
   *
   * @return bool
   *   TRUE if the rule applies to the path, FALSE otherwise.
   */
  public function appliesTo($current_path) {
    $configured_urls = $this->getPageUrl();

    // If no URLs configured, rule applies to all pages.
    if (empty($configured_urls)) {
      return TRUE;
    }

    // Normalize current path (remove trailing slashes).
    $current_path = rtrim($current_path, '/');

    // Handle root path.
    if (empty($current_path)) {
      $current_path = '/';
    }
    elseif (!str_starts_with($current_path, '/')) {
      $current_path = '/' . $current_path;
    }

    // Parse multiple URLs (one per line).
    $url_patterns = array_filter(array_map('trim', explode("\n", $configured_urls)));

    foreach ($url_patterns as $pattern) {
      // Normalize pattern (remove trailing slashes).
      $pattern = rtrim($pattern, '/');

      // Handle root path pattern.
      if (empty($pattern)) {
        $pattern = '/';
      }
      elseif (!str_starts_with($pattern, '/')) {
        $pattern = '/' . $pattern;
      }

      // Use Drupal's path matcher for consistent path matching.
      if ($this->getPathMatcher()->matchPath($current_path, $pattern)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Check if the given country should be restricted.
   *
   * @param string $country_code
   *   The country code to check.
   *
   * @return bool
   *   TRUE if the country should be restricted, FALSE otherwise.
   */
  public function isCountryRestricted($country_code) {
    $impacted_countries = $this->getImpactedCountries();
    switch ($this->getRestrictionType()) {
      case 'allow_except':
        // Allow all except selected countries.
        return in_array($country_code, $impacted_countries);

      case 'deny_except':
        // Deny all except selected countries.
        return !in_array($country_code, $impacted_countries);

      default:
        return FALSE;
    }
  }

}
