<?php

namespace Drupal\pagegeofence\EventSubscriber;

use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\pagegeofence\PageGeofenceManager;
use Drupal\pagegeofence\PageGeofenceWhitelistManager;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for page geofencing.
 */
class PageGeofenceSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * The page geofence manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceManager
   */
  protected $geofenceManager;

  /**
   * The whitelist manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceWhitelistManager
   */
  protected $whitelistManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The HTTP kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface
   */
  protected $httpKernel;

  /**
   * Constructs a PageGeofenceSubscriber object.
   *
   * @param \Drupal\pagegeofence\PageGeofenceManager $geofence_manager
   *   The page geofence manager.
   * @param \Drupal\pagegeofence\PageGeofenceWhitelistManager $whitelist_manager
   *   The whitelist manager.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Symfony\Component\HttpKernel\HttpKernelInterface $http_kernel
   *   The HTTP kernel.
   */
  public function __construct(PageGeofenceManager $geofence_manager, PageGeofenceWhitelistManager $whitelist_manager, AccountInterface $current_user, HttpKernelInterface $http_kernel) {
    $this->geofenceManager = $geofence_manager;
    $this->whitelistManager = $whitelist_manager;
    $this->currentUser = $current_user;
    $this->httpKernel = $http_kernel;
  }

  /**
   * {@inheritdoc}
   *
   * @return array
   *   An array of event names and their corresponding listener methods.
   */
  public static function getSubscribedEvents(): array {
    // Run early in the request cycle, but after routing.
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 50];
    return $events;
  }

  /**
   * Handles kernel request events.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The kernel request event.
   */
  public function onKernelRequest(RequestEvent $event) {
    // Only check main requests.
    if (!$event->isMainRequest()) {
      return;
    }

    $request = $event->getRequest();
    $current_path = $request->getPathInfo();

    // Skip admin routes and AJAX requests.
    if (strpos($current_path, '/admin') === 0 || $request->isXmlHttpRequest()) {
      return;
    }

    // Skip if user has permission to bypass geofencing.
    if ($this->currentUser->hasPermission('administer pagegeofence') ||
        $this->currentUser->hasPermission('bypass geofencing restrictions')) {
      return;
    }

    // Skip if the URL is whitelisted.
    if ($this->whitelistManager->isWhitelisted($current_path)) {
      return;
    }

    $user_country = $this->geofenceManager->getUserCountry();

    if ($user_country) {
      $restriction = $this->geofenceManager->isAccessRestricted($current_path, $user_country);
      if ($restriction) {
        // Check if a redirect URL is configured.
        $redirect_url = $restriction['redirect_url'];

        if (!empty($redirect_url)) {
          // Handle internal vs external URLs.
          if (strpos($redirect_url, '/') === 0) {
            // Internal URL - convert to absolute URL.
            $request = $event->getRequest();
            $base_url = $request->getSchemeAndHttpHost();
            $redirect_url = $base_url . $redirect_url;
          }

          // Log the restriction with redirect action.
          $this->geofenceManager->logActivity('Access restricted for country @country on path @path by rule @rule - redirected to @redirect_url', [
            '@country' => $user_country,
            '@path' => $current_path,
            '@rule' => $restriction['rule']->label(),
            '@redirect_url' => $redirect_url,
          ]);

          // Create a redirect response.
          $response = new Response('', 302, ['Location' => $redirect_url]);
          $event->setResponse($response);
        }
        else {
          // Log the restriction with 403 action.
          $this->geofenceManager->logActivity('Access restricted for country @country on path @path by rule @rule - showed 403 page', [
            '@country' => $user_country,
            '@path' => $current_path,
            '@rule' => $restriction['rule']->label(),
          ]);

          // Create a subrequest to render the restriction page with full theme.
          $response = $this->makeSubrequest($request, $restriction, $user_country);
          $event->setResponse($response);
        }
      }
    }
  }

  /**
   * Makes a subrequest to render the restriction page.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param array $restriction
   *   The restriction details.
   * @param string $country
   *   The user's country code.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response from the subrequest.
   */
  protected function makeSubrequest(Request $request, array $restriction, $country) {
    // Use Request::duplicate() to preserve all request context including
    // language negotiation, base URL, and other request attributes.
    $sub_request = $request->duplicate();

    // Override the path while preserving the original request context.
    $sub_request->server->set('REQUEST_URI', $request->getBaseUrl() . '/pagegeofence/restricted');
    $sub_request->server->set('SCRIPT_NAME', $request->getBaseUrl() . '/index.php');

    // Clear existing route attributes to force re-routing.
    $sub_request->attributes->remove('_route');
    $sub_request->attributes->remove('_route_params');
    $sub_request->attributes->remove('_controller');

    // Store restriction data and country in request attributes.
    $sub_request->attributes->set('pagegeofence_restriction', $restriction);
    $sub_request->attributes->set('pagegeofence_country', $this->geofenceManager->getCountryName($country));
    $sub_request->attributes->set('_is_subrequest', TRUE);

    // Handle the subrequest through the HTTP kernel.
    $response = $this->httpKernel->handle($sub_request, HttpKernelInterface::SUB_REQUEST);

    // Set the status code to 403.
    $response->setStatusCode(403);

    return $response;
  }

}
