<?php

namespace Drupal\pagegeofence\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Locale\CountryManagerInterface;
use Drupal\Core\Path\PathMatcherInterface;
use Drupal\Core\Url;
use Drupal\pagegeofence\PageGeofenceManager;
use Drupal\pagegeofence\PageGeofenceWhitelistManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a form for creating and editing page geofence rules.
 *
 * This form allows administrators to configure geographic restrictions for
 * specific pages or URL patterns, including restriction types, impacted
 * countries, and custom messaging for restricted users.
 */
class PageGeofenceRuleForm extends EntityForm {

  /**
   * The country manager service.
   *
   * @var \Drupal\Core\Locale\CountryManagerInterface
   */
  protected $countryManager;

  /**
   * The whitelist manager service.
   *
   * @var \Drupal\pagegeofence\PageGeofenceWhitelistManager
   */
  protected $whitelistManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The path matcher service.
   *
   * @var \Drupal\Core\Path\PathMatcherInterface
   */
  protected $pathMatcher;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The page geofence manager service.
   *
   * @var \Drupal\pagegeofence\PageGeofenceManager
   */
  protected $pageGeofenceManager;

  /**
   * Settings page url.
   *
   * @var string
   */
  protected $settingsUrl;

  /**
   * Constructs a PageGeofenceRuleForm object.
   *
   * @param \Drupal\Core\Locale\CountryManagerInterface $country_manager
   *   The country manager service.
   * @param \Drupal\pagegeofence\PageGeofenceWhitelistManager $whitelist_manager
   *   The whitelist manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Path\PathMatcherInterface $path_matcher
   *   The path matcher service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Drupal\pagegeofence\PageGeofenceManager $page_geofence_manager
   *   The page geofence manager service.
   */
  public function __construct(CountryManagerInterface $country_manager, PageGeofenceWhitelistManager $whitelist_manager, ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, PathMatcherInterface $path_matcher, RequestStack $request_stack, PageGeofenceManager $page_geofence_manager) {
    $this->countryManager = $country_manager;
    $this->whitelistManager = $whitelist_manager;
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->pathMatcher = $path_matcher;
    $this->requestStack = $request_stack;
    $this->pageGeofenceManager = $page_geofence_manager;
    $this->settingsUrl = Url::fromRoute('pagegeofence.settings')->toString();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('country_manager'),
      $container->get('pagegeofence.whitelist_manager'),
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('path.matcher'),
      $container->get('request_stack'),
      $container->get('pagegeofence.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\pagegeofence\Entity\PageGeofenceRule $rule */
    $rule = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Rule Name'),
      '#maxlength' => 255,
      '#default_value' => $rule->label(),
      '#description' => $this->t('A descriptive name for this geofencing rule.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $rule->id(),
      '#machine_name' => [
        'exists' => '\Drupal\pagegeofence\Entity\PageGeofenceRule::load',
      ],
      '#disabled' => !$rule->isNew(),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable this rule'),
      '#description' => $this->t('Uncheck to disable this rule without deleting it.'),
      '#default_value' => $rule->status(),
    ];

    $form['page_url'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Page URL Patterns'),
      '#description' => $this->t('Enter URL patterns to restrict, one per line. Leave empty to apply to all pages. Supports wildcards (*). Examples:<br>/career/award/secret-page<br>/restricted-section/*'),
      '#default_value' => $rule->getPageUrl(),
      '#placeholder' => 'e.g., /career/award/secret-page' . "\n" . '/restricted-section/*',
      '#required' => TRUE,
      '#rows' => 4,
    ];
    $form['rule_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Rule Type'),
      '#description' => $this->t('Select how you want to handle restricted access.'),
      '#options' => [
        'redirect_page' => $this->t('Redirect Page'),
        'show_message' => $this->t('Show Message'),
      ],
      '#default_value' => $this->getDefaultRuleType($rule),
      '#required' => TRUE,
    ];

    $form['restriction_content'] = [
      '#type' => 'details',
      '#title' => $this->t('Restriction Content'),
      '#open' => TRUE,
      '#description' => $this->t('Configure content for the restriction page.'),
      '#states' => [
        'visible' => [
          ':input[name="rule_type"]' => ['value' => 'show_message'],
        ],
      ],
    ];

    $form['restriction_content']['restriction_page_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Restriction Page Title'),
      '#description' => $this->t('The title to display on the restriction page.'),
      '#default_value' => $rule->getRestrictionPageTitle(),
      '#states' => [
        'required' => [
          ':input[name="rule_type"]' => ['value' => 'show_message'],
        ],
      ],
    ];

    $form['restriction_content']['restriction_page_summary'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Restriction Page Summary'),
      '#description' => $this->t('A brief summary to display on the restriction page.'),
      '#default_value' => $rule->getRestrictionPageSummary(),
      '#rows' => 3,
      '#states' => [
        'visible' => [
          ':input[name="rule_type"]' => ['value' => 'show_message'],
        ],
      ],
    ];
    // Check for global settings to indicate overrides.
    $default_reason_format = filter_default_format();
    if (!empty($rule->getReasonForRestriction()['format'])) {
      $default_reason_format = $rule->getReasonForRestriction()['format'];
    }
    $form['restriction_content']['reason_for_restriction'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Reason for Restriction'),
      '#description' => $this->t('Provide the legal or business reason behind the restriction (supports rich text formatting).'),
      '#default_value' => $rule->getReasonForRestriction()['value'] ?? '',
      '#format' => $default_reason_format,
      '#states' => [
        'required' => [
          ':input[name="rule_type"]' => ['value' => 'show_message'],
        ],
      ],
    ];

    $form['restriction_content']['country_information_flag'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Country Information'),
      '#description' => $this->t('Enable to display additional country-specific information on the restriction page.'),
      '#default_value' => $rule->getCountryInformationFlag(),
    ];

    $form['restriction_content']['country_information_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Country Information Text'),
      '#description' => $this->t('Additional information to display when country information is enabled.'),
      '#default_value' => $rule->getCountryInformationText() ?? 'Country Origin:',
      '#rows' => 4,
      '#states' => [
        'visible' => [
          ':input[name="country_information_flag"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['redirect_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Redirect URL'),
      '#description' => $this->t('Optional: Enter a URL to redirect users to instead of showing the default 403 page.<br><strong>Internal:</strong> /path/to/page<br><strong>External:</strong> https://example.com/blocked<br>Leave empty to use the default 403 restriction page. Cannot be the same as restricted URL patterns.'),
      '#default_value' => $rule->getRedirectUrl(),
      '#placeholder' => '/restricted-notice or https://example.com/blocked',
      '#states' => [
        'visible' => [
          ':input[name="rule_type"]' => ['value' => 'redirect_page'],
        ],
      ],
    ];



    $form['restriction_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Restriction Type'),
      '#description' => $this->t('Choose how the geofencing should be applied.'),
      '#options' => [
        'allow_except' => $this->t('Allow all except selected countries'),
        'deny_except' => $this->t('Deny all except selected countries'),
      ],
      '#default_value' => $rule->getRestrictionType() ?: 'allow_except',
      '#required' => TRUE,
    ];

    $form['impacted_countries'] = [
      '#type' => 'select',
      '#title' => $this->t('Impacted Countries'),
      '#description' => $this->t('Select the countries that will be impacted by this restriction. At least one country must be selected.'),
      '#options' => $this->countryManager->getList(),
      '#default_value' => $rule->getImpactedCountries(),
      '#multiple' => TRUE,
      '#size' => 10,
      '#required' => TRUE,
    ];

    // Display current country detected from headers.
    $detected_country = $this->pageGeofenceManager->getUserCountry();
    $country_list = $this->countryManager->getList();

    if ($detected_country) {
      $country_name = $country_list[$detected_country] ?? $detected_country;
      $form['current_country_display'] = [
        '#type' => 'item',
        '#title' => $this->t('Currently Detected Country'),
        '#markup' => '<div class="messages messages--status">' .
        $this->t('Your current location is detected as: <strong>@country (@code)</strong>', [
          '@country' => $country_name,
          '@code' => $detected_country,
        ]) . '</div>',
      ];
    }
    else {
      $form['current_country_display'] = [
        '#type' => 'item',
        '#title' => $this->t('Country Detection Status'),
        '#markup' => '<div class="messages messages--error">' .
        $this->t('No country could be detected from the current request headers. Please check your country detection configuration in the <a href="@settings_url">PageGeofence settings</a>.', [
          '@settings_url' => $this->settingsUrl,
        ]) . '</div>',
      ];
    }

    $form['weight'] = [
      '#type' => 'number',
      '#title' => $this->t('Weight'),
      '#description' => $this->t('Rules with lower weights are processed first. Use this to control the order of rule evaluation.'),
      '#default_value' => $rule->getWeight(),
      '#step' => 1,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate that country header is configured.
    $config = $this->configFactory->get('pagegeofence.settings');
    $country_header = $config->get('country_header');
    if (empty($country_header) || !is_string($country_header)) {
      $form_state->setError($form,
        $this->t('Country detection header is not configured. Please <a href="@settings_url">configure a country header</a> before creating geofence rules.', [
          '@settings_url' => $this->settingsUrl,
        ]));
      return;
    }

    // Validate rule_type: must be selected.
    $rule_type = $form_state->getValue('rule_type');
    if (empty($rule_type)) {
      $form_state->setErrorByName('rule_type',
        $this->t('Please select a rule type (Redirect Page or Show Message).'));
      return;
    }

    $show_content_enabled = ($rule_type === 'show_content');
    $redirect_page_enabled = ($rule_type === 'redirect_page');

    // Validate restriction content fields only if show_content is enabled.
    if ($show_content_enabled) {
      // Validate restriction_page_title: required.
      $title = trim($form_state->getValue('restriction_page_title'));
      if (empty($title)) {
        $form_state->setErrorByName('restriction_page_title',
          $this->t('Restriction page title is required when "Show Message" is enabled.'));
      }

      // Validate reason for restriction minimum length.
      $reason = $form_state->getValue('reason_for_restriction');
      $reason_text = is_array($reason) ? trim($reason['value']) : trim($reason);

      // Reason is required with minimum 50 characters.
      if (empty($reason_text)) {
        $form_state->setErrorByName('reason_for_restriction',
          $this->t('Reason for restriction is required when "Show Message" is enabled.'));
      }

      // Validate country_information_text if flag is checked and no global
      // setting.
      $country_info_flag = $form_state->getValue('country_information_flag');
      $country_info_text = trim($form_state->getValue('country_information_text'));
      if ($country_info_flag && !$has_global_country_text && empty($country_info_text)) {
        $form_state->setErrorByName('country_information_text',
          $this->t('Country information text is required when "Show Message" is enabled. You can set a global default in the <a href="@settings_url">PageGeofence settings</a> to make this field optional.', [
            '@settings_url' => $this->settingsUrl,
          ]));
      }
    }

    // Validate that at least one country is selected.
    $countries = $form_state->getValue('impacted_countries');
    if (empty($countries) || (is_array($countries) && count(array_filter($countries)) === 0)) {
      $form_state->setErrorByName('impacted_countries',
        $this->t('At least one country must be selected.'));
    }

    // Validate page URL format if provided.
    $page_urls = trim($form_state->getValue('page_url'));
    if (!empty($page_urls)) {
      $urls = array_filter(array_map('trim', explode("\n", $page_urls)));
      foreach ($urls as $line_number => $url) {
        if (!preg_match('/^\//', $url)) {
          $form_state->setErrorByName('page_url',
            $this->t('Each URL must start with a forward slash (/). Error on line @line: @url', [
              '@line' => $line_number + 1,
              '@url' => $url,
            ]));
          break;
        }
        // Validate wildcard usage - only allow * at the end or as a complete segment.
        if (strpos($url, '*') !== FALSE) {
          if (!preg_match('/^\/[^*]*(\*|\/\*)?$/', $url)) {
            $form_state->setErrorByName('page_url',
              $this->t('Invalid wildcard usage on line @line: @url. Wildcards (*) can only be used at the end of a path or as complete path segments.', [
                '@line' => $line_number + 1,
                '@url' => $url,
              ]));
            break;
          }
        }
      }

      // Check for conflicts with whitelist URLs.
      $conflicts = $this->whitelistManager->checkForConflicts($page_urls);
      if (!empty($conflicts)) {
        $conflict_messages = [];
        foreach ($conflicts as $conflict) {
          $conflict_messages[] = $this->t('@pattern conflicts with whitelisted pattern @whitelist', [
            '@pattern' => $conflict['pattern'],
            '@whitelist' => $conflict['whitelist_pattern'],
          ]);
        }
        $form_state->setErrorByName('page_url',
          $this->t('The following URL patterns conflict with whitelisted URLs and cannot be geo-restricted:<br>@conflicts<br><a href="@settings_url">Manage whitelist settings</a>', [
            '@conflicts' => implode('<br>', $conflict_messages),
            '@settings_url' => $this->settingsUrl,
          ]));
      }

      // Check for duplicate or conflicting URL patterns with existing rules.
      $restriction_type = $form_state->getValue('restriction_type');
      $rule_conflicts = $this->checkUrlConflicts($page_urls, $restriction_type);

      // Handle exact duplicates.
      if (!empty($rule_conflicts['duplicates'])) {
        $duplicate_messages = [];
        foreach ($rule_conflicts['duplicates'] as $duplicate) {
          $duplicate_messages[] = $this->t('@pattern is already used in rule "@rule"', [
            '@pattern' => $duplicate['pattern'],
            '@rule' => $duplicate['existing_rule'],
          ]);
        }
        $form_state->setErrorByName('page_url',
          $this->t('The following URL patterns are already in use by other rules:<br>@duplicates<br>Each URL pattern can only be used once across all geofence rules.', [
            '@duplicates' => implode('<br>', $duplicate_messages),
          ]));
      }

      // Handle overlapping patterns with same restriction type.
      if (!empty($rule_conflicts['overlaps'])) {
        $overlap_messages = [];
        foreach ($rule_conflicts['overlaps'] as $overlap) {
          $overlap_messages[] = $this->t('@pattern overlaps with "@existing_pattern" from rule "@rule" (both use "@restriction_type")', [
            '@pattern' => $overlap['pattern'],
            '@existing_pattern' => $overlap['existing_pattern'],
            '@rule' => $overlap['existing_rule'],
            '@restriction_type' => $overlap['restriction_type'] === 'allow_except' ? 'Allow all except selected' : 'Deny all except selected',
          ]);
        }
        $form_state->setErrorByName('page_url',
          $this->t('The following URL patterns overlap with existing rules that have the same restriction type:<br>@overlaps<br>Overlapping patterns must have different restriction types to avoid conflicts.', [
            '@overlaps' => implode('<br>', $overlap_messages),
          ]));
      }
    }

    // Validate redirect URL if provided.
    $redirect_url = trim($form_state->getValue('redirect_url'));
    if (!empty($redirect_url)) {
      // Check if it's internal URL (starts with /) or external URL.
      $is_internal = strpos($redirect_url, '/') === 0;

      if ($is_internal) {
        // For internal URLs, validate format.
        if (!preg_match('/^\//', $redirect_url)) {
          $form_state->setErrorByName('redirect_url',
            $this->t('Internal redirect URL must start with a forward slash (/).'));
        }

        // Check that internal redirect URL is not the same as any page_url
        // patterns.
        $page_urls = trim($form_state->getValue('page_url'));
        if (!empty($page_urls)) {
          $url_patterns = array_filter(array_map('trim', explode("\n", $page_urls)));

          // Normalize redirect URL.
          $normalized_redirect = rtrim($redirect_url, '/');
          if (empty($normalized_redirect)) {
            $normalized_redirect = '/';
          }

          foreach ($url_patterns as $pattern) {
            // Normalize pattern.
            $normalized_pattern = rtrim($pattern, '/');
            if (empty($normalized_pattern)) {
              $normalized_pattern = '/';
            }

            // Check for exact match or if redirect URL matches the pattern.
            if ($normalized_redirect === $normalized_pattern ||
                $this->pathMatcher->matchPath($normalized_redirect, $normalized_pattern)) {
              $form_state->setErrorByName('redirect_url',
                $this->t('Redirect URL "@redirect" cannot be the same as or match the restricted URL pattern "@pattern". This would create a redirect loop.', [
                  '@redirect' => $redirect_url,
                  '@pattern' => $pattern,
                ]));
              break;
            }
          }
        }
      }
      else {
        // For external URLs, validate format.
        if (!filter_var($redirect_url, FILTER_VALIDATE_URL)) {
          $form_state->setErrorByName('redirect_url',
            $this->t('Please enter a valid URL. For external URLs use http:// or https://, for internal URLs start with /.'));
        }
        // Recommend HTTPS for external URLs.
        elseif (parse_url($redirect_url, PHP_URL_SCHEME) !== 'https') {
          $this->messenger()->addWarning(
            $this->t('For security reasons, it is recommended to use HTTPS URLs for external redirects.')
          );
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\pagegeofence\Entity\PageGeofenceRule $rule */
    $rule = $this->entity;

    // Get the selected rule type.
    $rule_type = $form_state->getValue('rule_type');

    // Clear fields based on rule type to prevent saving conflicting data.
    if ($rule_type === 'redirect_page') {
      // Clear "Show Message" fields when "Redirect Page" is selected.
      $rule->setRestrictionPageTitle('');
      $rule->setRestrictionPageSummary('');
      $rule->set('reason_for_restriction', [
        'value' => '',
        'format' => filter_default_format(),
      ]);
      $rule->setCountryInformationFlag(FALSE);
      $rule->setCountryInformationText('');
      
      // Set redirect URL.
      $rule->setRedirectUrl($form_state->getValue('redirect_url'));
    }
    elseif ($rule_type === 'show_message') {
      // Clear "Redirect Page" fields when "Show Message" is selected.
      $rule->setRedirectUrl('');
      
      // Set "Show Message" fields.
      $rule->setRestrictionPageTitle($form_state->getValue('restriction_page_title'));
      $rule->setRestrictionPageSummary($form_state->getValue('restriction_page_summary'));
      
      // Handle text_format field for reason_for_restriction.
      $reason = $form_state->getValue('reason_for_restriction');
      $rule->set('reason_for_restriction', [
        'value' => $reason['value'],
        'format' => $reason['format'],
      ]);
      
      $rule->setCountryInformationFlag($form_state->getValue('country_information_flag'));
      $rule->setCountryInformationText($form_state->getValue('country_information_text'));
    }

    // Set common fields regardless of rule type.
    $rule->setPageUrl($form_state->getValue('page_url'));
    $rule->setRestrictionType($form_state->getValue('restriction_type'));
    $rule->setImpactedCountries($form_state->getValue('impacted_countries'));
    $rule->setWeight($form_state->getValue('weight'));

    $status = $rule->save();

    switch ($status) {
      case SAVED_NEW:
        $this->messenger()->addMessage($this->t('Created the %label page geofence rule.', [
          '%label' => $rule->label(),
        ]));
        break;

      default:
        $this->messenger()->addMessage($this->t('Saved the %label page geofence rule.', [
          '%label' => $rule->label(),
        ]));
    }

    $form_state->setRedirectUrl($rule->toUrl('collection'));
  }

  /**
   * Check for duplicate or conflicting URL patterns with existing rules.
   *
   * @param string $url_patterns
   *   URL patterns to check (newline separated).
   * @param string $restriction_type
   *   The restriction type of the current rule.
   *
   * @return array
   *   Array of conflicts with keys: 'duplicates' and 'overlaps'.
   */
  protected function checkUrlConflicts($url_patterns, $restriction_type) {
    $rule = $this->entity;
    $current_rule_id = FALSE;
    if (!$rule->isNew()) {
      $current_rule_id = $rule->id();
    }
    $storage = $this->entityTypeManager->getStorage('pagegeofence_rule');
    $existing_rules = $storage->loadMultiple();

    // Remove current rule from check if editing.
    if ($current_rule_id && isset($existing_rules[$current_rule_id])) {
      unset($existing_rules[$current_rule_id]);
    }

    $patterns = array_filter(array_map('trim', explode("\n", $url_patterns)));
    $duplicates = [];
    $overlaps = [];

    foreach ($patterns as $pattern) {
      // Normalize pattern.
      $pattern = rtrim($pattern, '/');
      if (empty($pattern)) {
        $pattern = '/';
      }
      elseif (!str_starts_with($pattern, '/')) {
        $pattern = '/' . $pattern;
      }

      foreach ($existing_rules as $rule) {
        /** @var \Drupal\pagegeofence\Entity\PageGeofenceRule $rule */
        if (!$rule->status()) {
          // Skip disabled rules.
          continue;
        }

        $existing_urls = $rule->getPageUrl();
        $existing_patterns = array_filter(array_map('trim', explode("\n", $existing_urls)));

        foreach ($existing_patterns as $existing_pattern) {
          // Normalize existing pattern.
          $existing_pattern = rtrim($existing_pattern, '/');
          if (empty($existing_pattern)) {
            $existing_pattern = '/';
          }
          elseif (!str_starts_with($existing_pattern, '/')) {
            $existing_pattern = '/' . $existing_pattern;
          }

          // Check for exact duplicates.
          if ($pattern === $existing_pattern) {
            $duplicates[] = [
              'pattern' => $pattern,
              'existing_rule' => $rule->label(),
              'existing_rule_id' => $rule->id(),
            ];
          }
          // Check for overlapping patterns using PathMatcher.
          elseif ($this->pathMatcher->matchPath($pattern, $existing_pattern) ||
                  $this->pathMatcher->matchPath($existing_pattern, $pattern)) {
            // If patterns overlap, restriction types should be different.
            if ($restriction_type === $rule->getRestrictionType()) {
              $overlaps[] = [
                'pattern' => $pattern,
                'existing_pattern' => $existing_pattern,
                'existing_rule' => $rule->label(),
                'existing_rule_id' => $rule->id(),
                'restriction_type' => $restriction_type,
              ];
            }
          }
        }
      }
    }

    return ['duplicates' => $duplicates, 'overlaps' => $overlaps];
  }

  /**
   * Get default rule type based on existing rule configuration.
   *
   * @param \Drupal\pagegeofence\Entity\PageGeofenceRule $rule
   *   The rule entity.
   *
   * @return array
   *   Array of checked rule types.
   */
  protected function getDefaultRuleType($rule) {
    // If redirect URL is set, default to redirect_page.
    if (!empty($rule->getRedirectUrl())) {
      return 'redirect_page';
    }

    // If any restriction content fields are set, default to show_content.
    if (!empty($rule->getRestrictionPageTitle()) ||
        !empty($rule->getRestrictionPageSummary()) ||
        !empty($rule->getReasonForRestriction()['value']) ||
        !empty($rule->getCountryInformationText())) {
      return 'show_message';
    }

    // Default to show_message for new rules.
    return 'show_message';
  }

}
