<?php

namespace Drupal\pagegeofence;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Locale\CountryManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for managing page geofencing functionality.
 */
class PageGeofenceManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The country manager.
   *
   * @var \Drupal\Core\Locale\CountryManagerInterface
   */
  protected $countryManager;

  /**
   * Constructs a PageGeofenceManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Locale\CountryManagerInterface $country_manager
   *   The country manager.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, RequestStack $request_stack, LoggerChannelFactoryInterface $logger_factory, CountryManagerInterface $country_manager) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
    $this->loggerFactory = $logger_factory;
    $this->countryManager = $country_manager;
  }

  /**
   * Check if the current request should be restricted based on geofencing.
   *
   * @param string $current_path
   *   The current path to check.
   * @param string $user_country
   *   The user's country code (ISO 3166-1 alpha-2).
   *
   * @return array|false
   *   Array with restriction details if access should be restricted
   *   FALSE otherwise.
   *   Array contains:
   *     'restricted' => TRUE,
   *     'page_title' => string,
   *     'restriction_summary' => string,
   *     'reason' => string,
   *     'rule' => PageGeofenceRule,
   *     'country_info_flag' => string,
   *     'country_info_text' => string,
   *     'redirect_url' => string|null
   */
  public function isAccessRestricted($current_path, $user_country) {
    $rules = $this->getActiveRules();
    foreach ($rules as $rule) {
      if ($rule->appliesTo($current_path) && $rule->isCountryRestricted($user_country)) {
        // Use rule values directly.
        $page_title = $rule->getRestrictionPageTitle();
        $restriction_summary = $rule->getRestrictionPageSummary();
        $reason = $rule->getFormattedReasonForRestriction();
        $country_info_flag = $rule->getCountryInformationFlag();
        $country_info_text = $rule->getCountryInformationText();
        $redirect_url = $rule->getRedirectUrl();

        $result = [
          'restricted' => TRUE,
          'rule' => $rule,
        ];

        if (!empty($page_title)) {
          $result['page_title'] = $page_title;
        }
        if (!empty($restriction_summary)) {
          $result['restriction_summary'] = $restriction_summary;
        }
        if (!empty($reason)) {
          $result['reason'] = $reason;
        }
        if (!empty($country_info_flag)) {
          $result['country_info_flag'] = $country_info_flag;
        }
        if (!empty($country_info_text)) {
          $result['country_info_text'] = $country_info_text;
        }
        if (!empty($redirect_url)) {
          $result['redirect_url'] = $redirect_url;
        }
        return $result;
      }
    }

    return FALSE;
  }

  /**
   * Get all active geofence rules ordered by weight.
   *
   * @return \Drupal\pagegeofence\Entity\PageGeofenceRule[]
   *   Array of active page geofence rules.
   */
  public function getActiveRules() {
    $storage = $this->entityTypeManager->getStorage('pagegeofence_rule');

    $rules = $storage->loadByProperties(['status' => TRUE]);

    // Sort by weight (lowest first).
    uasort($rules, function ($a, $b) {
      return $a->getWeight() <=> $b->getWeight();
    });
    return $rules;
  }

  /**
   * Get the user's country code from the request.
   *
   * Method checks configured headers in order to determine the user's country.
   *
   * @return string
   *   The user's country code (ISO 3166-1 alpha-2).
   */
  public function getUserCountry() {
    $request = $this->requestStack->getCurrentRequest();
    $return = FALSE;
    // Get configured country header.
    $config = $this->configFactory->get('pagegeofence.settings');
    $country_header = $config->get('country_header') ?? '';
    // Check the configured header.
    $country_code = $request->headers->get($country_header);
    if ($country_code && strlen($country_code) === 2) {
      $return = strtoupper($country_code);
    }

    return $return;
  }

  /**
   * Log geofencing activity.
   *
   * @param string $message
   *   The message to log.
   * @param array $context
   *   Additional context for the log entry.
   */
  public function logActivity($message, array $context = []) {
    // Only log if logging is enabled in configuration.
    $config = $this->configFactory->get('pagegeofence.settings');
    if (!($config->get('enable_logging') ?? TRUE)) {
      return;
    }

    $this->loggerFactory->get('pagegeofence')->info($message, $context);
  }

  /**
   * Get the restriction reason from the first matching rule.
   *
   * @param string $current_path
   *   The current path to check.
   * @param string $user_country
   *   The user's country code.
   *
   * @return string
   *   The restriction reason, or empty string if no restriction applies.
   */
  public function getRestrictionReason($current_path, $user_country) {
    $restriction = $this->isAccessRestricted($current_path, $user_country);
    return $restriction ? $restriction['reason'] : '';
  }

  /**
   * Get the country name from a country code.
   *
   * @param string $country_code
   *   The two-letter ISO 3166-1 alpha-2 country code.
   *
   * @return string
   *   The country name, or the country code if not found.
   */
  public function getCountryName($country_code) {
    if (empty($country_code)) {
      return '';
    }

    $countries = $this->countryManager->getList();
    $country_code = strtoupper($country_code);

    return $countries[$country_code] ?? $country_code;
  }

}
