<?php

namespace Drupal\Tests\pagegeofence\Unit;

use Drupal\pagegeofence\PageGeofenceManager;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for PageGeofenceManager service.
 *
 * @coversDefaultClass \Drupal\pagegeofence\PageGeofenceManager
 * @group pagegeofence
 */
class PageGeofenceManagerTest extends UnitTestCase {

  /**
   * The page geofence manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceManager
   */
  protected $manager;

  /**
   * Mock config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $requestStack;

  /**
   * Mock logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $loggerFactory;

  /**
   * Mock country manager.
   *
   * @var \Drupal\Core\Locale\CountryManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $countryManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mocks.
    $this->configFactory = $this->createMock('\Drupal\Core\Config\ConfigFactoryInterface');
    $this->entityTypeManager = $this->createMock('\Drupal\Core\Entity\EntityTypeManagerInterface');
    $this->requestStack = $this->createMock('\Symfony\Component\HttpFoundation\RequestStack');
    $this->loggerFactory = $this->createMock('\Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $this->countryManager = $this->createMock('\Drupal\Core\Locale\CountryManagerInterface');

    // Create the manager.
    $this->manager = new PageGeofenceManager(
      $this->configFactory,
      $this->entityTypeManager,
      $this->requestStack,
      $this->loggerFactory,
      $this->countryManager
    );
  }

  /**
   * Tests getUserCountry method returns country from header.
   *
   * @covers ::getUserCountry
   */
  public function testGetUserCountryFromHeader() {
    $request = $this->createMock('\Symfony\Component\HttpFoundation\Request');
    $request->headers = $this->createMock('\Symfony\Component\HttpFoundation\HeaderBag');

    $config = $this->createMock('\Drupal\Core\Config\ImmutableConfig');
    $config->method('get')
      ->with('country_header')
      ->willReturn('CF-IPCountry');

    $this->configFactory->method('get')
      ->with('pagegeofence.settings')
      ->willReturn($config);

    $this->requestStack->method('getCurrentRequest')
      ->willReturn($request);

    $request->headers->method('get')
      ->with('CF-IPCountry')
      ->willReturn('us');

    $result = $this->manager->getUserCountry();
    $this->assertEquals('US', $result);
  }

  /**
   * Tests getCountryName method returns country name.
   *
   * @covers ::getCountryName
   */
  public function testGetCountryName() {
    $this->countryManager->method('getList')
      ->willReturn([
        'US' => 'United States',
        'DE' => 'Germany',
        'FR' => 'France',
      ]);

    $result = $this->manager->getCountryName('US');
    $this->assertEquals('United States', $result);

    $result = $this->manager->getCountryName('DE');
    $this->assertEquals('Germany', $result);

    // Test unknown country code.
    $result = $this->manager->getCountryName('XX');
    $this->assertEquals('XX', $result);

    // Test empty country code.
    $result = $this->manager->getCountryName('');
    $this->assertEquals('', $result);
  }

  /**
   * Tests logActivity when logging is enabled.
   *
   * @covers ::logActivity
   */
  public function testLogActivityWhenEnabled() {
    $config = $this->createMock('\Drupal\Core\Config\ImmutableConfig');
    $logger = $this->createMock('\Psr\Log\LoggerInterface');

    $config->method('get')
      ->with('enable_logging')
      ->willReturn(TRUE);

    $configFactory = $this->createMock('\Drupal\Core\Config\ConfigFactoryInterface');
    $configFactory->method('get')
      ->with('pagegeofence.settings')
      ->willReturn($config);

    $loggerFactory = $this->createMock('\Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $loggerFactory->method('get')
      ->with('pagegeofence')
      ->willReturn($logger);

    $logger->expects($this->once())
      ->method('info')
      ->with('Test message');

    $manager = new PageGeofenceManager(
      $configFactory,
      $this->entityTypeManager,
      $this->requestStack,
      $loggerFactory,
      $this->countryManager
    );

    $manager->logActivity('Test message');
  }

  /**
   * Tests logActivity when logging is disabled.
   *
   * @covers ::logActivity
   */
  public function testLogActivityWhenDisabled() {
    $config = $this->createMock('\Drupal\Core\Config\ImmutableConfig');
    $logger = $this->createMock('\Psr\Log\LoggerInterface');

    $config->method('get')
      ->with('enable_logging')
      ->willReturn(FALSE);

    $configFactory = $this->createMock('\Drupal\Core\Config\ConfigFactoryInterface');
    $configFactory->method('get')
      ->with('pagegeofence.settings')
      ->willReturn($config);

    $loggerFactory = $this->createMock('\Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $loggerFactory->method('get')
      ->with('pagegeofence')
      ->willReturn($logger);

    $logger->expects($this->never())
      ->method('info');

    $manager = new PageGeofenceManager(
      $configFactory,
      $this->entityTypeManager,
      $this->requestStack,
      $loggerFactory,
      $this->countryManager
    );

    $manager->logActivity('Test message');
  }

}
