<?php

namespace Drupal\Tests\pagegeofence\Unit;

use Drupal\pagegeofence\Controller\PageGeofenceRestrictionController;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * Unit tests for PageGeofenceRestrictionController.
 *
 * @coversDefaultClass \Drupal\pagegeofence\Controller\PageGeofenceRestrictionController
 * @group pagegeofence
 */
class PageGeofenceRestrictionControllerTest extends UnitTestCase {

  /**
   * The controller under test.
   *
   * @var \Drupal\pagegeofence\Controller\PageGeofenceRestrictionController|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $controller;

  /**
   * Mock request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $requestStack;

  /**
   * Mock request.
   *
   * @var \Symfony\Component\HttpFoundation\Request|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $request;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->requestStack = $this->createMock(RequestStack::class);
    $this->request = $this->createMock(Request::class);
    $this->request->attributes = $this->createMock(ParameterBag::class);

    $this->requestStack->method('getCurrentRequest')
      ->willReturn($this->request);

    // Create a partial mock to override the t() method.
    $this->controller = $this->getMockBuilder(PageGeofenceRestrictionController::class)
      ->setConstructorArgs([$this->requestStack])
      ->onlyMethods(['t'])
      ->getMock();

    // Mock the t() method to return the input string.
    $this->controller->method('t')
      ->willReturnCallback(function ($string) {
        return $string;
      });
  }

  /**
   * Tests getTitle returns restriction page title.
   *
   * @covers ::getTitle
   */
  public function testGetTitleReturnsRestrictionTitle() {
    $restrictionData = [
      'page_title' => 'Access Denied',
      'restriction_summary' => 'This content is not available.',
      'reason' => 'Geographic restriction',
      'country_info_flag' => TRUE,
      'country_info_text' => 'US visitors cannot access this.',
      'rule' => 'test_rule',
    ];

    $this->request->attributes->method('get')
      ->with('pagegeofence_restriction')
      ->willReturn($restrictionData);

    $title = $this->controller->getTitle();
    $this->assertEquals('Access Denied', $title);
  }

  /**
   * Tests getTitle returns default title when no restriction data.
   *
   * @covers ::getTitle
   */
  public function testGetTitleReturnsDefaultWhenNoData() {
    $this->request->attributes->method('get')
      ->with('pagegeofence_restriction')
      ->willReturn(NULL);

    $title = $this->controller->getTitle();
    $this->assertEquals('Access Restricted', $title);
  }

  /**
   * Tests access is allowed when restriction data is present.
   *
   * @covers ::access
   */
  public function testAccessAllowedWithRestrictionData() {
    $restrictionData = [
      'page_title' => 'Restricted',
      'restriction_summary' => 'Summary',
      'reason' => 'Reason',
      'country_info_flag' => TRUE,
      'country_info_text' => 'Info',
      'rule' => 'test_rule',
    ];

    $this->request->attributes->method('get')
      ->willReturnCallback(function ($key) use ($restrictionData) {
        if ($key === 'pagegeofence_restriction') {
          return $restrictionData;
        }
        if ($key === 'pagegeofence_country') {
          return 'US';
        }
        return NULL;
      });

    $result = $this->controller->access();
    $this->assertTrue($result->isAllowed());
  }

  /**
   * Tests access is forbidden when no restriction data.
   *
   * @covers ::access
   */
  public function testAccessForbiddenWithoutRestrictionData() {
    $this->request->attributes->method('get')
      ->willReturn(NULL);

    $result = $this->controller->access();
    $this->assertTrue($result->isForbidden());
  }

  /**
   * Tests access is forbidden when only restriction data exists (no country).
   *
   * @covers ::access
   */
  public function testAccessForbiddenWithoutCountryData() {
    $restrictionData = [
      'page_title' => 'Restricted',
      'restriction_summary' => 'Summary',
      'reason' => 'Reason',
      'country_info_flag' => TRUE,
      'country_info_text' => 'Info',
      'rule' => 'test_rule',
    ];

    $this->request->attributes->method('get')
      ->willReturnCallback(function ($key) use ($restrictionData) {
        if ($key === 'pagegeofence_restriction') {
          return $restrictionData;
        }
        return NULL;
      });

    $result = $this->controller->access();
    $this->assertTrue($result->isForbidden());
  }

  /**
   * Tests restrictionPage returns proper render array.
   *
   * @covers ::restrictionPage
   */
  public function testRestrictionPageReturnsRenderArray() {
    $restrictionData = [
      'page_title' => 'Content Blocked',
      'restriction_summary' => 'This page is restricted in your region.',
      'reason' => 'Geographic restrictions apply.',
      'country_info_flag' => TRUE,
      'country_info_text' => 'United States visitors cannot view this content.',
      'rule' => 'us_restriction_rule',
    ];

    $this->request->attributes->method('get')
      ->willReturnCallback(function ($key) use ($restrictionData) {
        if ($key === 'pagegeofence_restriction') {
          return $restrictionData;
        }
        if ($key === 'pagegeofence_country') {
          return 'US';
        }
        return NULL;
      });

    $result = $this->controller->restrictionPage();

    $this->assertIsArray($result);
    $this->assertEquals('pagegeofence_restriction_page', $result['#theme']);
    $this->assertEquals('Content Blocked', $result['#page_title']);
    $this->assertEquals('This page is restricted in your region.', $result['#restriction_summary']);
    $this->assertEquals('Geographic restrictions apply.', $result['#reason']);
    $this->assertEquals('US', $result['#country']);
    $this->assertTrue($result['#country_info_flag']);
    $this->assertEquals('United States visitors cannot view this content.', $result['#country_info_text']);
    $this->assertEquals('us_restriction_rule', $result['#rule']);
  }

  /**
   * Tests restrictionPage handles empty restriction data.
   *
   * @covers ::restrictionPage
   */
  public function testRestrictionPageWithEmptyData() {
    $emptyRestrictionData = [
      'page_title' => '',
      'restriction_summary' => '',
      'reason' => '',
      'country_info_flag' => FALSE,
      'country_info_text' => '',
      'rule' => '',
    ];

    $this->request->attributes->method('get')
      ->willReturnCallback(function ($key) use ($emptyRestrictionData) {
        if ($key === 'pagegeofence_restriction') {
          return $emptyRestrictionData;
        }
        if ($key === 'pagegeofence_country') {
          return '';
        }
        return NULL;
      });

    $result = $this->controller->restrictionPage();

    $this->assertIsArray($result);
    $this->assertEquals('pagegeofence_restriction_page', $result['#theme']);
    $this->assertEmpty($result['#page_title']);
    $this->assertEmpty($result['#restriction_summary']);
    $this->assertEmpty($result['#reason']);
    $this->assertEmpty($result['#country']);
    $this->assertFalse($result['#country_info_flag']);
  }

}
