<?php

namespace Drupal\Tests\pagegeofence\Unit;

use Drupal\pagegeofence\Controller\PageGeofenceRuleController;
use Drupal\pagegeofence\Entity\PageGeofenceRule;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Unit tests for PageGeofenceRuleController.
 *
 * @coversDefaultClass \Drupal\pagegeofence\Controller\PageGeofenceRuleController
 * @group pagegeofence
 */
class PageGeofenceRuleControllerTest extends UnitTestCase {

  /**
   * The controller under test.
   *
   * @var \Drupal\pagegeofence\Controller\PageGeofenceRuleController
   */
  protected $controller;

  /**
   * Mock messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * Mock string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $stringTranslation;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->stringTranslation = $this->createMock(TranslationInterface::class);

    // Create a partial mock that allows us to test the enable/disable methods.
    $this->controller = $this->getMockBuilder(PageGeofenceRuleController::class)
      ->onlyMethods(['messenger', 't', 'redirect'])
      ->getMock();

    $this->controller->method('messenger')
      ->willReturn($this->messenger);

    // Mock the t() method to return the input string.
    $this->controller->method('t')
      ->willReturnCallback(function ($string, $args = []) {
        return strtr($string, $args);
      });
  }

  /**
   * Tests enable method enables rule and shows message.
   *
   * @covers ::enable
   */
  public function testEnableRuleEnablesAndShowsMessage() {
    $rule = $this->createMock(PageGeofenceRule::class);
    
    $rule->expects($this->once())
      ->method('setStatus')
      ->with(TRUE);
    
    $rule->expects($this->once())
      ->method('save');
    
    $rule->method('label')
      ->willReturn('Test Rule');

    $this->messenger->expects($this->once())
      ->method('addMessage')
      ->with('The page geofence rule Test Rule has been enabled.');

    $redirectResponse = $this->createMock(RedirectResponse::class);
    $this->controller->method('redirect')
      ->with('entity.pagegeofence_rule.collection')
      ->willReturn($redirectResponse);

    $result = $this->controller->enable($rule);
    $this->assertInstanceOf(RedirectResponse::class, $result);
  }

  /**
   * Tests disable method disables rule and shows message.
   *
   * @covers ::disable
   */
  public function testDisableRuleDisablesAndShowsMessage() {
    $rule = $this->createMock(PageGeofenceRule::class);
    
    $rule->expects($this->once())
      ->method('setStatus')
      ->with(FALSE);
    
    $rule->expects($this->once())
      ->method('save');
    
    $rule->method('label')
      ->willReturn('Test Rule');

    $this->messenger->expects($this->once())
      ->method('addMessage')
      ->with('The page geofence rule Test Rule has been disabled.');

    $redirectResponse = $this->createMock(RedirectResponse::class);
    $this->controller->method('redirect')
      ->with('entity.pagegeofence_rule.collection')
      ->willReturn($redirectResponse);

    $result = $this->controller->disable($rule);
    $this->assertInstanceOf(RedirectResponse::class, $result);
  }

  /**
   * Tests enable handles rule with special characters in label.
   *
   * @covers ::enable
   */
  public function testEnableHandlesSpecialCharactersInLabel() {
    $rule = $this->createMock(PageGeofenceRule::class);
    
    $rule->method('setStatus');
    $rule->method('save');
    $rule->method('label')
      ->willReturn('Test "Rule" & <Special>');

    $this->messenger->expects($this->once())
      ->method('addMessage')
      ->with('The page geofence rule Test "Rule" & <Special> has been enabled.');

    $redirectResponse = $this->createMock(RedirectResponse::class);
    $this->controller->method('redirect')
      ->willReturn($redirectResponse);

    $result = $this->controller->enable($rule);
    $this->assertInstanceOf(RedirectResponse::class, $result);
  }

  /**
   * Tests disable handles rule with special characters in label.
   *
   * @covers ::disable
   */
  public function testDisableHandlesSpecialCharactersInLabel() {
    $rule = $this->createMock(PageGeofenceRule::class);
    
    $rule->method('setStatus');
    $rule->method('save');
    $rule->method('label')
      ->willReturn('Test "Rule" & <Special>');

    $this->messenger->expects($this->once())
      ->method('addMessage')
      ->with('The page geofence rule Test "Rule" & <Special> has been disabled.');

    $redirectResponse = $this->createMock(RedirectResponse::class);
    $this->controller->method('redirect')
      ->willReturn($redirectResponse);

    $result = $this->controller->disable($rule);
    $this->assertInstanceOf(RedirectResponse::class, $result);
  }

  /**
   * Tests enable method redirects to collection route.
   *
   * @covers ::enable
   */
  public function testEnableRedirectsToCollection() {
    $rule = $this->createMock(PageGeofenceRule::class);
    $rule->method('setStatus');
    $rule->method('save');
    $rule->method('label')->willReturn('Test');

    $this->messenger->method('addMessage');

    $redirectResponse = $this->createMock(RedirectResponse::class);
    
    $this->controller->expects($this->once())
      ->method('redirect')
      ->with('entity.pagegeofence_rule.collection')
      ->willReturn($redirectResponse);

    $this->controller->enable($rule);
  }

  /**
   * Tests disable method redirects to collection route.
   *
   * @covers ::disable
   */
  public function testDisableRedirectsToCollection() {
    $rule = $this->createMock(PageGeofenceRule::class);
    $rule->method('setStatus');
    $rule->method('save');
    $rule->method('label')->willReturn('Test');

    $this->messenger->method('addMessage');

    $redirectResponse = $this->createMock(RedirectResponse::class);
    
    $this->controller->expects($this->once())
      ->method('redirect')
      ->with('entity.pagegeofence_rule.collection')
      ->willReturn($redirectResponse);

    $this->controller->disable($rule);
  }

}
