<?php

namespace Drupal\Tests\pagegeofence\Unit;

use Drupal\Core\Session\AccountInterface;
use Drupal\pagegeofence\EventSubscriber\PageGeofenceSubscriber;
use Drupal\pagegeofence\PageGeofenceManager;
use Drupal\pagegeofence\PageGeofenceWhitelistManager;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Unit tests for PageGeofenceSubscriber.
 *
 * @coversDefaultClass \Drupal\pagegeofence\EventSubscriber\PageGeofenceSubscriber
 * @group pagegeofence
 */
class PageGeofenceSubscriberTest extends UnitTestCase {

  /**
   * The page geofence subscriber.
   *
   * @var \Drupal\pagegeofence\EventSubscriber\PageGeofenceSubscriber
   */
  protected $subscriber;

  /**
   * Mock page geofence manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $geofenceManager;

  /**
   * Mock whitelist manager.
   *
   * @var \Drupal\pagegeofence\PageGeofenceWhitelistManager|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $whitelistManager;

  /**
   * Mock current user.
   *
   * @var \Drupal\Core\Session\AccountInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $currentUser;

  /**
   * Mock HTTP kernel.
   *
   * @var \Symfony\Component\HttpKernel\HttpKernelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $httpKernel;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->geofenceManager = $this->createMock(PageGeofenceManager::class);
    $this->whitelistManager = $this->createMock(PageGeofenceWhitelistManager::class);
    $this->currentUser = $this->createMock(AccountInterface::class);
    $this->httpKernel = $this->createMock(HttpKernelInterface::class);

    $this->subscriber = new PageGeofenceSubscriber(
      $this->geofenceManager,
      $this->whitelistManager,
      $this->currentUser,
      $this->httpKernel
    );
  }

  /**
   * Tests that subrequests are skipped.
   *
   * @covers ::onKernelRequest
   */
  public function testSkipsSubrequests() {
    $request = Request::create('/test-page');
    $request->attributes->set('_is_subrequest', TRUE);

    $event = new RequestEvent(
      $this->httpKernel,
      $request,
      HttpKernelInterface::MAIN_REQUEST
    );

    // Manager should not be called for subrequests.
    $this->geofenceManager->expects($this->never())
      ->method('isAccessRestricted');

    $this->subscriber->onKernelRequest($event);

    // Event should not have a response set.
    $this->assertFalse($event->hasResponse());
  }

  /**
   * Tests that admin routes are skipped.
   *
   * @covers ::onKernelRequest
   */
  public function testSkipsAdminRoutes() {
    $request = Request::create('/admin/config');
    $request->attributes->set('_route', 'system.admin_config');

    $event = new RequestEvent(
      $this->httpKernel,
      $request,
      HttpKernelInterface::MAIN_REQUEST
    );

    // Manager should not be called for admin routes.
    $this->geofenceManager->expects($this->never())
      ->method('isAccessRestricted');

    $this->subscriber->onKernelRequest($event);

    $this->assertFalse($event->hasResponse());
  }

  /**
   * Tests that restriction page itself is skipped.
   *
   * @covers ::onKernelRequest
   */
  public function testSkipsRestrictionPage() {
    $request = Request::create('/pagegeofence/restricted');

    $event = new RequestEvent(
      $this->httpKernel,
      $request,
      HttpKernelInterface::MAIN_REQUEST
    );

    // Manager should not be called for restriction page.
    $this->geofenceManager->expects($this->never())
      ->method('isAccessRestricted');

    $this->subscriber->onKernelRequest($event);

    $this->assertFalse($event->hasResponse());
  }

  /**
   * Tests language switch detection strips language prefix.
   *
   * @covers ::onKernelRequest
   */
  public function testLanguageSwitchDetection() {
    $session = $this->createMock(SessionInterface::class);

    // Simulate visiting /about, then switching to /de/about.
    $request = Request::create('/de/about');
    $request->setSession($session);
    $request->headers->set('referer', 'http://example.com/about');

    $event = new RequestEvent(
      $this->httpKernel,
      $request,
      HttpKernelInterface::MAIN_REQUEST
    );

    // This would normally trigger language switch logic.
    // In a real scenario, the session would be updated.
    // For unit tests, we can only verify the logic doesn't error.
    $this->geofenceManager->method('isAccessRestricted')
      ->willReturn(FALSE);

    $this->whitelistManager->method('isWhitelisted')
      ->willReturn(FALSE);

    // Should not throw an exception.
    $this->subscriber->onKernelRequest($event);

    $this->assertFalse($event->hasResponse());
  }

  /**
   * Tests that unrestricted pages don't trigger restriction.
   *
   * @covers ::onKernelRequest
   */
  public function testUnrestrictedPageAllowed() {
    $request = Request::create('/public-page');
    $session = $this->createMock(SessionInterface::class);
    $request->setSession($session);

    $event = new RequestEvent(
      $this->httpKernel,
      $request,
      HttpKernelInterface::MAIN_REQUEST
    );

    // Page is not restricted.
    $this->geofenceManager->method('isAccessRestricted')
      ->willReturn(FALSE);

    $this->whitelistManager->method('isWhitelisted')
      ->willReturn(FALSE);

    $this->subscriber->onKernelRequest($event);

    // No response should be set.
    $this->assertFalse($event->hasResponse());
  }

  /**
   * Tests getSubscribedEvents returns correct events.
   *
   * @covers ::getSubscribedEvents
   */
  public function testGetSubscribedEvents() {
    $events = PageGeofenceSubscriber::getSubscribedEvents();

    $this->assertIsArray($events);
    $this->assertArrayHasKey('kernel.request', $events);
    $this->assertIsArray($events['kernel.request']);
    $this->assertCount(1, $events['kernel.request']);
    $this->assertEquals(['onKernelRequest', 50], $events['kernel.request'][0]);
  }

}
