<?php

namespace Drupal\pagetree\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The Page tree configuration form.
 *
 * @package Drupal\pagetree\Form
 */
class ConfigurationForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManager
   */
  protected $entityTypeManager;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManager
   */
  protected $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->languageManager = $container->get('language_manager');

    return $instance;
  }

  /**
   * {@inheritDoc}
   */
  protected function getEditableConfigNames() {
    return [
      'pagetree.settings',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'pagetree';
  }

  /**
   * {@inheritDoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $values = $this->configFactory()->get($this->getEditableConfigNames()[0]);

    $form['general'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('General settings'),
      '#description' => $this->t('Set the general settings for the page tree modules.'),
    ];

    // Let the user pick the menus to display.
    $menus = $this->entityTypeManager->getStorage('menu')->loadMultiple();
    $menusList = [];
    foreach ($menus as $menu) {
      $menusList[$menu->id()] = $menu->label();
    }
    asort($menusList);

    $form['general']['menus'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Menus'),
      '#description' => $this->t('Select the menus to be displayed in the page tree.'),
      '#options' => $menusList,
      '#default_value' => $values->get('menus'),
    ];

    // Let the user pick the languages to display.
    $languages = $this->languageManager->getLanguages();
    $languagesList = [];
    foreach ($languages as $language) {
      $languagesList[$language->getId()] = $language->getName();
    }
    asort($languagesList);

    $form['general']['languages'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Languages'),
      '#description' => $this->t('Select the languages to be displayed in the page tree.'),
      '#options' => $languagesList,
      '#default_value' => $values->get('languages'),
    ];

    // Let the user pick the content types to display.
    $contentTypes = $this->entityTypeManager->getStorage('node_type')->loadMultiple();
    $contentTypesList = [];
    foreach ($contentTypes as $contentType) {
      $contentTypesList[$contentType->id()] = $contentType->label();
    }
    $form['general']['contenttypes'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Content types'),
      '#description' => $this->t('Select the content types you want to process with the page tree module.'),
      '#options' => $contentTypesList,
      '#default_value' => $values->get('contentTypes'),
    ];

    $form['general']['reset_status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Set unpublished?'),
      '#description' => $this->t('If checked, the publication status of the clone will be unpublished.'),
      '#default_value' => $values->get('reset_status'),
    ];

    $titleSourceList = [
      'node' => $this->t('Node'),
      'menu_link' => $this->t('Menu link'),
    ];
    $form['general']['title_source'] = [
      '#type' => 'select',
      '#title' => 'Pagetree title source',
      '#description' => $this->t('Select the source of the titles, it can be taken from the node or menu links.'),
      '#options' => $titleSourceList,
      '#default_value' => !empty($values->get('title_source')) ? $values->get('title_source') : 'node',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritDoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    parent::submitForm($form, $form_state);
    // Store set values.
    $this->configFactory()->getEditable(self::getEditableConfigNames()[0])
      ->set('menus', array_filter($form_state->getValue('menus')))
      ->set('languages', array_filter($form_state->getValue('languages')))
      ->set('contentTypes', array_filter($form_state->getValue('contenttypes')))
      ->set('reset_status', $form_state->getValue('reset_status'))
      ->set('title_source', $form_state->getValue('title_source'))
      ->save();
  }

}
