/**
 * @file
 * Defines the behavior of the palette entity browser view.
 *
 * Behavior:
 * - Shows hex input + "Select entities" button initially
 * - After selecting a color:
 *   - Hides the hex input
 *   - Shows the color preview circle + label
 *   - Keeps the browser button visible (to change color later)
 */

(function ($, Drupal, once) {

  'use strict';

  /**
   * Updates the visual style of a row based on checkbox state.
   */
  function updateClasses($row, $input) {
    var isChecked = $input.prop('checked');
    var label = $row.find('.views-field-label').text().trim();
    var colorHex = $row.find('.views-field-rgb').text().trim();

    if (label && colorHex) {
      $row.attr('title', label + ' (' + colorHex + ')');
    }

    var styles = {
      'background-color': colorHex || '#eee',
      'border': '3px solid ' + (isChecked ? '#007bff' : 'transparent'),
      'border-radius': '50%',
      'width': '50px',
      'height': '50px',
      'display': 'inline-block',
      'margin': '10px',
      'cursor': 'pointer',
      'transition': 'transform 0.2s ease, border-color 0.2s ease',
      'box-shadow': isChecked ? '0 0 8px rgba(0,123,255,0.6)' : '0 2px 4px rgba(0,0,0,0.1)',
      'position': 'relative',
      'overflow': 'hidden',
      'transform': isChecked ? 'scale(1.1)' : 'scale(1)'
    };

    $row.css(styles);
    $row.find('> div').css('display', 'none');
  }

  Drupal.behaviors.paletteEntityBrowserView = {
    attach: function (context) {
      var $viewContent = $('.view-id-palette .view-content', context);
      var $viewRows = $('.views-row', context);

      if ($viewContent.length) {
        $viewContent.css({
          'display': 'flex',
          'flex-wrap': 'wrap',
          'justify-content': 'flex-start'
        });
      }

      $viewRows.each(function () {
        var $row = $(this);
        var $input = $row.find('input[name^="entity_browser_select"]');
        if ($input.length) {
          updateClasses($row, $input);
        }
      });

      // Single selection on click
      $(once('palette-selection', '.views-row', context)).on('click', function (e) {
        if ($(e.target).is('input')) return;

        var $row = $(this);
        var $input = $row.find('input[name^="entity_browser_select"]');

        if ($input.length) {
          var shouldCheck = !$input.prop('checked');

          // Deselect all others
          $viewRows.each(function () {
            var $otherRow = $(this);
            var $otherInput = $otherRow.find('input[name^="entity_browser_select"]');
            if ($otherInput.length) {
              $otherInput.prop('checked', false);
              updateClasses($otherRow, $otherInput);
            }
          });

          $input.prop('checked', shouldCheck);
          updateClasses($row, $input);
          $input.trigger('change');
        }
      });

      // Hover effect
      $viewRows.hover(
        function () {
          if (!$(this).find('input[name^="entity_browser_select"]').prop('checked')) {
            $(this).css('transform', 'scale(1.1)');
          }
        },
        function () {
          if (!$(this).find('input[name^="entity_browser_select"]').prop('checked')) {
            $(this).css('transform', 'scale(1)');
          }
        }
      );

      // When a color is selected → update parent form UI
      $(once('palette-update-parent', 'input[name^="entity_browser_select"]', context))
        .on('change', function () {
          var $input = $(this);
          if ($input.prop('checked')) {
            var $row = $input.closest('.views-row');
            var colorHex = $row.find('.views-field-rgb').text().trim();
            var colorLabel = $row.find('.views-field-label').text().trim();

            if (colorHex) {
              var $parentDoc = $(window.parent.document);
              var $widget = $parentDoc.find('.palette-color-widget-container');

              if ($widget.length) {
                // Hide hex input
                $widget.find('.palette-color-input-wrapper')
                  .css({
                    'position': 'absolute',
                    'left': '-9999px',
                    'width': '1px',
                    'height': '1px',
                    'overflow': 'hidden'
                  });

                // Show preview + label
                $widget.find('.palette-selected-info').show();

                // Update preview circle
                $widget.find('.palette-color-preview').css('background-color', colorHex);

                // Update label (use name if available, fallback to hex)
                $widget.find('.palette-color-label').text(colorLabel || colorHex);

                // IMPORTANT: Keep the hex value in the hidden input so it gets saved
                $widget.find('input.palette-color-input').val(colorHex);
              }
            }
          }
        });
    }
  };

}(jQuery, Drupal, once));
