<?php

declare(strict_types=1);

namespace Drupal\palette\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'palette_color_widget' widget.
 *
 * Behavior:
 * - Initially shows hex input + Entity Browser button
 * - After selecting a color: hides hex input and shows color preview circle + label
 * - Allows manual hex entry or selection via browser
 *
 * @FieldWidget(
 *   id = "palette_color_widget",
 *   label = @Translation("Palette Color Browser"),
 *   field_types = {"color_field_type"}
 * )
 */
final class PaletteColorWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $value = $items[$delta]->color ?? '';

    // Unique wrapper ID per delta
    $wrapper_id = 'palette-color-widget-wrapper-' . $delta;

    // Main container
    $element = [
      '#type' => 'container',
      '#attributes' => [
        'id' => $wrapper_id,
        'class' => ['palette-color-widget-container'],
      ],
      '#prefix' => '<div style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap; margin-bottom: 10px;">',
      '#suffix' => '</div>',
    ];

    // Selected color preview + label (hidden initially if no value, shown after JS selection)
    $element['selected_info'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['palette-selected-info'],
        'style' => $value ? '' : 'display: none;',
      ],
    ];

    $element['selected_info']['preview'] = [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['palette-color-preview'],
        'style' => 'background-color:' . ($value ?: '#ccc') . '; width:50px; height:50px; border-radius:50%; border:3px solid #007bff; flex-shrink:0;',
      ],
    ];

    $element['selected_info']['label'] = [
      '#type' => 'markup',
      '#markup' => '<span class="palette-color-label">' . ($value ? htmlspecialchars($value) : 'Color seleccionado') . '</span>',
    ];

    // Hex input: always present initially, will be hidden by JS after selection
    $element['color_input_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['palette-color-input-wrapper']],
    ];

    $element['color_input_wrapper']['color'] = [
      '#type' => 'textfield',
      '#title' => $element['#title'] ?? '',
      '#default_value' => $value,
      '#size' => 10,
      '#maxlength' => 7,
      '#attributes' => [
        'class' => ['palette-color-input'],
        'placeholder' => '#000000',
      ],
      '#wrapper_attributes' => [
        'class' => ['palette-color-input-wrapper'],
      ],
    ];

    // Entity Browser button (always visible)
    $element['browser'] = [
      '#type' => 'entity_browser',
      '#entity_browser' => 'palette',
      '#cardinality' => 1,
      '#selection_mode' => \Drupal\entity_browser\Element\EntityBrowserElement::SELECTION_MODE_APPEND,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   *
   * Processes the submitted values before they are saved.
   * Since the textfield is nested inside ['color_input_wrapper']['color'],
   * Drupal won't automatically map it to the 'color' property.
   * We manually extract and normalize the hex value here.
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state): array {
    foreach ($values as &$value) {
      // The submitted value comes from $element['color_input_wrapper']['color']
      $submitted_hex = $value['color_input_wrapper']['color'] ?? '';

      // Clean and normalize the hex value
      $hex = trim($submitted_hex);

      if ($hex !== '') {
        // Accept formats: #fff, #ffffff, fff, ffffff, #FFF, etc.
        if (preg_match('/^#?[0-9A-Fa-f]{3,6}$/i', $hex)) {
          // Add leading # if missing
          if ($hex[0] !== '#') {
            $hex = '#' . $hex;
          }

          // Expand shorthand (#fff → #ffffff)
          if (strlen($hex) === 4) {
            $hex = '#' . $hex[1] . $hex[1] . $hex[2] . $hex[2] . $hex[3] . $hex[3];
          }

          // Force uppercase
          $value['color'] = strtoupper($hex);
        } else {
          // Invalid hex format → store empty (or you could add validation error)
          $value['color'] = '';
        }
      } else {
        $value['color'] = '';
      }

      // Optional: clean up the temporary nested structure
      unset($value['color_input_wrapper']);
    }

    return $values;
  }

}
