<?php

declare(strict_types=1);

namespace Drupal\palette\Plugin\EntityBrowser\Widget;

use Drupal\Core\Form\FormStateInterface;
use Drupal\entity_browser\WidgetBase;
use Drupal\palette\Entity\PaletteColor;

/**
 * Provides a widget to add palette colors within the entity browser.
 *
 * @EntityBrowserWidget(
 *   id = "palette_color_add",
 *   label = @Translation("Add Palette Color"),
 *   description = @Translation("Adds a form to create and immediately select a new palette color."),
 *   auto_select = TRUE
 * )
 */
final class PaletteColorAddWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return array_merge(parent::defaultConfiguration(), [
      'submit_text' => $this->t('Save and select color'),
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getForm(array &$original_form, FormStateInterface $form_state, array $additional_widget_parameters): array {
    $form = parent::getForm($original_form, $form_state, $additional_widget_parameters);

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Color Name'),
      '#required' => TRUE,
      '#placeholder' => $this->t('e.g. Corporate Blue'),
    ];

    $form['rgb'] = [
      '#type' => 'textfield',
      '#title' => $this->t('RGB Color'),
      '#required' => TRUE,
      '#default_value' => '#',
      '#attributes' => ['placeholder' => '#RRGGBB'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function prepareEntities(array $form, FormStateInterface $form_state): array {
    $color = PaletteColor::create([
      'label' => $form_state->getValue(['label']),
      'rgb' => $form_state->getValue(['rgb']),
      'status' => TRUE,
    ]);

    return [$color];
  }

  /**
   * {@inheritdoc}
   */
  public function submit(array &$element, array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getTriggeringElement()['#eb_widget_main_submit'])) {
      $colors = $this->prepareEntities($form, $form_state);

      foreach ($colors as $color) {
        $color->save();
      }

      $this->selectEntities($colors, $form_state);
      $this->clearFormValues($element, $form_state);
    }
  }

  /**
   * Clears form values after submission.
   */
  protected function clearFormValues(array &$element, FormStateInterface $form_state): void {
    $form_state->setValueForElement($element['label'], '');
    $form_state->setValueForElement($element['rgb'], '');
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['submit_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Submit button text'),
      '#default_value' => $this->configuration['submit_text'],
    ];
    return $form;
  }

}
