<?php

declare(strict_types=1);

namespace Drupal\pantheon_si_tokens\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Pantheon SI Tokens settings.
 */
class SettingsForm extends ConfigFormBase
{

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array
  {
    return ['pantheon_si_tokens.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string
  {
    return 'pantheon_si_tokens_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $config = $this->config('pantheon_si_tokens.settings');
    $constants = $config->get('constants') ?? [];

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => $this->t(
        'Configure which Pantheon Secure Integration PHP ' .
        'constants to expose as tokens. Constants should be in ' .
        'the format PANTHEON_SOIP_SERVICENAME.',
      ),
    ];

    $form['constants'] = [
      '#type' => 'textarea',
      '#title' => $this->t('PHP Constants'),
      '#description' => $this->t(
        'Enter one PHP constant name per line. Example: PANTHEON_SOIP_LDAP',
      ),
      '#default_value' => implode("\n", $constants),
      '#rows' => 10,
    ];

    $form['token_help'] = [
      '#type' => 'markup',
      '#markup' => $this->t(
        '<p><strong>Token format:</strong> Constants will be ' .
        'available as <code>[pantheon_si_tunnel:' .
        'constant_name_lowercase]</code></p><p>Example: ' .
        '<code>PANTHEON_SOIP_LDAP</code> becomes ' .
        '<code>[pantheon_si_tunnel:pantheon_soip_ldap]</code></p>',
      ),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    parent::validateForm($form, $form_state);

    $constants_text = $form_state->getValue('constants');
    $constants = array_filter(
      array_map('trim', explode("\n", $constants_text)),
    );

    foreach ($constants as $constant) {
      if (!str_starts_with($constant, 'PANTHEON_SOIP_')) {
        $form_state->setErrorByName(
          'constants',
          $this->t(
            'Constant "@constant" must start with "PANTHEON_SOIP_". ' .
            'This module only exposes Pantheon Secure Integration ' .
            'tunnel constants.',
            ['@constant' => $constant],
          ),
        );
        continue;
      }

      if (!defined($constant)) {
        $form_state->setErrorByName(
          'constants',
          $this->t(
            'Constant "@constant" is not defined. Ensure Pantheon ' .
            'Secure Integration is configured and the constant name ' .
            'is spelled correctly.',
            ['@constant' => $constant],
          ),
        );
        continue;
      }

      $value = constant($constant);

      if (!is_int($value) && !ctype_digit((string) $value)) {
        $form_state->setErrorByName(
          'constants',
          $this->t(
            'Constant "@constant" has value "@value" which is not ' .
            'an integer. Pantheon SI tunnel constants must contain ' .
            'port numbers.',
            [
              '@constant' => $constant,
              '@value' => is_scalar($value) ? $value : gettype($value),
            ],
          ),
        );
        continue;
      }

      $port = (int) $value;
      if ($port < 1 || $port > 65535) {
        $form_state->setErrorByName(
          'constants',
          $this->t(
            'Constant "@constant" has value @port which is outside ' .
            'the valid port range (1-65535).',
            [
              '@constant' => $constant,
              '@port' => $port,
            ],
          ),
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    $constants_text = $form_state->getValue('constants');
    $constants = array_filter(
      array_map('trim', explode("\n", $constants_text)),
    );

    $this->config('pantheon_si_tokens.settings')
      ->set('constants', array_values($constants))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
