<?php

declare(strict_types=1);

namespace Drupal\Tests\pantheon_si_tokens\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Pantheon SI Tokens settings form.
 *
 * @group pantheon_si_tokens
 */
class SettingsFormTest extends BrowserTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'pantheon_si_tokens',
    'token',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests the settings form access.
   */
  public function testSettingsFormAccess(): void
  {
    $this->drupalGet('admin/config/system/pantheon-si-tokens');
    $this->assertSession()->statusCodeEquals(403);

    $admin_user = $this->drupalCreateUser([
      'administer site configuration',
    ]);
    $this->drupalLogin($admin_user);

    $this->drupalGet('admin/config/system/pantheon-si-tokens');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains(
      'Configure which Pantheon Secure Integration PHP constants',
    );
  }

  /**
   * Tests form renders and saves empty configuration.
   */
  public function testFormRendersAndSavesEmpty(): void
  {
    $admin_user = $this->drupalCreateUser([
      'administer site configuration',
    ]);
    $this->drupalLogin($admin_user);

    $this->drupalGet('admin/config/system/pantheon-si-tokens');
    $this->submitForm(
      ['constants' => ''],
      'Save configuration',
    );

    $this->assertSession()->pageTextContains(
      'The configuration options have been saved',
    );

    $config = $this->config('pantheon_si_tokens.settings');
    $saved_constants = $config->get('constants');

    $this->assertIsArray($saved_constants);
    $this->assertCount(0, $saved_constants);
  }

  /**
   * Tests validation rejects non-PANTHEON_SOIP constants.
   */
  public function testValidationRejectsInvalidPrefix(): void
  {
    $admin_user = $this->drupalCreateUser([
      'administer site configuration',
    ]);
    $this->drupalLogin($admin_user);

    $this->drupalGet('admin/config/system/pantheon-si-tokens');
    $this->submitForm(
      ['constants' => 'INVALID_CONSTANT'],
      'Save configuration',
    );

    $this->assertSession()->pageTextContains(
      'Constant "INVALID_CONSTANT" must start with "PANTHEON_SOIP_"',
    );
    $this->assertSession()->pageTextContains(
      'This module only exposes Pantheon Secure Integration',
    );
  }

  /**
   * Tests validation rejects undefined constants.
   */
  public function testValidationRejectsUndefinedConstants(): void
  {
    $admin_user = $this->drupalCreateUser([
      'administer site configuration',
    ]);
    $this->drupalLogin($admin_user);

    $this->drupalGet('admin/config/system/pantheon-si-tokens');
    $this->submitForm(
      ['constants' => 'PANTHEON_SOIP_UNDEFINED_CONSTANT'],
      'Save configuration',
    );

    $this->assertSession()->pageTextContains('is not defined');
    $this->assertSession()->pageTextContains(
      'Ensure Pantheon Secure Integration is configured',
    );
  }

}
