<?php

declare(strict_types=1);

namespace Drupal\Tests\pantheon_si_tokens\Kernel;

use Drupal\Core\Form\FormState;
use Drupal\KernelTests\KernelTestBase;
use Drupal\pantheon_si_tokens\Form\SettingsForm;

/**
 * Tests SettingsForm validation logic.
 *
 * @group pantheon_si_tokens
 */
class SettingsFormValidationTest extends KernelTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'pantheon_si_tokens',
    'system',
  ];

  /**
   * The settings form.
   *
   * @var \Drupal\pantheon_si_tokens\Form\SettingsForm
   */
  protected SettingsForm $settingsForm;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();
    $this->installConfig(['pantheon_si_tokens']);
    $this->settingsForm = SettingsForm::create($this->container);
  }

  /**
   * Tests validation rejects non-PANTHEON_SOIP constants.
   */
  public function testValidationRejectsInvalidPrefix(): void
  {
    $form = [];
    $form_state = new FormState();
    $form_state->setValue('constants', 'INVALID_CONSTANT');

    $this->settingsForm->validateForm($form, $form_state);

    $errors = $form_state->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString(
      'must start with "PANTHEON_SOIP_"',
      (string) $errors['constants'],
    );
  }

  /**
   * Tests validation rejects non-integer values.
   */
  public function testValidationRejectsNonIntegerValues(): void
  {
    if (!defined('PANTHEON_SOIP_STRING_VALUE')) {
      define('PANTHEON_SOIP_STRING_VALUE', 'not_a_number');
    }

    $form = [];
    $form_state = new FormState();
    $form_state->setValue('constants', 'PANTHEON_SOIP_STRING_VALUE');

    $this->settingsForm->validateForm($form, $form_state);

    $errors = $form_state->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString(
      'which is not an integer',
      (string) $errors['constants'],
    );
  }

  /**
   * Tests validation rejects port 0.
   */
  public function testValidationRejectsPortZero(): void
  {
    if (!defined('PANTHEON_SOIP_PORT_ZERO')) {
      define('PANTHEON_SOIP_PORT_ZERO', 0);
    }

    $form = [];
    $form_state = new FormState();
    $form_state->setValue('constants', 'PANTHEON_SOIP_PORT_ZERO');

    $this->settingsForm->validateForm($form, $form_state);

    $errors = $form_state->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString(
      'outside the valid port range (1-65535)',
      (string) $errors['constants'],
    );
  }

  /**
   * Tests validation rejects port above 65535.
   */
  public function testValidationRejectsPortTooHigh(): void
  {
    if (!defined('PANTHEON_SOIP_PORT_TOO_HIGH')) {
      define('PANTHEON_SOIP_PORT_TOO_HIGH', 65536);
    }

    $form = [];
    $form_state = new FormState();
    $form_state->setValue('constants', 'PANTHEON_SOIP_PORT_TOO_HIGH');

    $this->settingsForm->validateForm($form, $form_state);

    $errors = $form_state->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString(
      'outside the valid port range (1-65535)',
      (string) $errors['constants'],
    );
  }

  /**
   * Tests validation accepts valid ports.
   */
  public function testValidationAcceptsValidPorts(): void
  {
    if (!defined('PANTHEON_SOIP_VALID_PORT_1')) {
      define('PANTHEON_SOIP_VALID_PORT_1', 1);
    }
    if (!defined('PANTHEON_SOIP_VALID_PORT_33061')) {
      define('PANTHEON_SOIP_VALID_PORT_33061', 33061);
    }
    if (!defined('PANTHEON_SOIP_VALID_PORT_65535')) {
      define('PANTHEON_SOIP_VALID_PORT_65535', 65535);
    }

    $form = [];
    $form_state = new FormState();
    $form_state->setValue(
      'constants',
      "PANTHEON_SOIP_VALID_PORT_1\n" .
      "PANTHEON_SOIP_VALID_PORT_33061\n" .
      "PANTHEON_SOIP_VALID_PORT_65535",
    );

    $this->settingsForm->validateForm($form, $form_state);

    $errors = $form_state->getErrors();
    $this->assertCount(0, $errors);
  }

  /**
   * Tests validation rejects undefined constants.
   */
  public function testValidationRejectsUndefinedConstants(): void
  {
    $form = [];
    $form_state = new FormState();
    $form_state->setValue('constants', 'PANTHEON_SOIP_NOT_DEFINED');

    $this->settingsForm->validateForm($form, $form_state);

    $errors = $form_state->getErrors();
    $this->assertCount(1, $errors);
    $this->assertStringContainsString(
      'is not defined',
      (string) $errors['constants'],
    );
    $this->assertStringContainsString(
      'Ensure Pantheon Secure Integration is configured',
      (string) $errors['constants'],
    );
  }

}
