<?php

declare(strict_types=1);

namespace Drupal\Tests\pantheon_si_tokens\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Tests Pantheon SI token integration.
 *
 * @group pantheon_si_tokens
 */
class TokenIntegrationTest extends KernelTestBase
{

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'pantheon_si_tokens',
    'token',
    'system',
    'user',
  ];

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void
  {
    parent::setUp();
    $this->installConfig(['pantheon_si_tokens', 'system']);
    $this->token = $this->container->get('token');
  }

  /**
   * Tests token info registration.
   */
  public function testTokenInfoRegistration(): void
  {
    $config = $this->config('pantheon_si_tokens.settings');
    $config->set(
      'constants',
      ['PANTHEON_SOIP_LDAP', 'PANTHEON_SOIP_DATABASE'],
    )->save();

    $token_info = $this->token->getInfo();

    $this->assertArrayHasKey('pantheon_si_tunnel', $token_info['types']);
    $this->assertEquals(
      'Pantheon SI Tunnel',
      $token_info['types']['pantheon_si_tunnel']['name']->__toString(),
    );

    $this->assertArrayHasKey(
      'pantheon_si_tunnel',
      $token_info['tokens'],
    );
    $this->assertArrayHasKey(
      'pantheon_soip_ldap',
      $token_info['tokens']['pantheon_si_tunnel'],
    );
    $this->assertArrayHasKey(
      'pantheon_soip_database',
      $token_info['tokens']['pantheon_si_tunnel'],
    );

    $this->assertArrayNotHasKey(
      'PANTHEON_SOIP_LDAP',
      $token_info['tokens']['pantheon_si_tunnel'],
      'Token names should be lowercase',
    );
  }

  /**
   * Tests token replacement with defined constant.
   */
  public function testTokenReplacementWithDefinedConstant(): void
  {
    if (!defined('PANTHEON_SOIP_TEST')) {
      define('PANTHEON_SOIP_TEST', '12345');
    }

    $config = $this->config('pantheon_si_tokens.settings');
    $config->set('constants', ['PANTHEON_SOIP_TEST'])->save();

    $text = 'Port: [pantheon_si_tunnel:pantheon_soip_test]';
    $replaced = $this->token->replace($text);

    $this->assertEquals('Port: 12345', $replaced);
  }

  /**
   * Tests token replacement with undefined constant.
   */
  public function testTokenReplacementWithUndefinedConstant(): void
  {
    $config = $this->config('pantheon_si_tokens.settings');
    $config->set('constants', ['PANTHEON_SOIP_UNDEFINED'])->save();

    $text = 'Port: [pantheon_si_tunnel:pantheon_soip_undefined]';
    $replaced = $this->token->replace($text);

    $this->assertEquals('Port: ', $replaced);
  }

  /**
   * Tests token replacement with unconfigured constant.
   */
  public function testTokenReplacementWithUnconfiguredConstant(): void
  {
    if (!defined('PANTHEON_SOIP_UNCONFIGURED')) {
      define('PANTHEON_SOIP_UNCONFIGURED', '99999');
    }

    $config = $this->config('pantheon_si_tokens.settings');
    $config->set('constants', [])->save();

    $text = 'Port: [pantheon_si_tunnel:pantheon_soip_unconfigured]';
    $replaced = $this->token->replace($text);

    $this->assertEquals($text, $replaced);
  }

  /**
   * Tests multiple token replacement.
   */
  public function testMultipleTokenReplacement(): void
  {
    if (!defined('PANTHEON_SOIP_LDAP')) {
      define('PANTHEON_SOIP_LDAP', '33061');
    }
    if (!defined('PANTHEON_SOIP_DATABASE')) {
      define('PANTHEON_SOIP_DATABASE', '33062');
    }

    $config = $this->config('pantheon_si_tokens.settings');
    $config->set(
      'constants',
      ['PANTHEON_SOIP_LDAP', 'PANTHEON_SOIP_DATABASE'],
    )->save();

    $text = 'LDAP: [pantheon_si_tunnel:pantheon_soip_ldap], ' .
      'DB: [pantheon_si_tunnel:pantheon_soip_database]';
    $replaced = $this->token->replace($text);

    $this->assertEquals('LDAP: 33061, DB: 33062', $replaced);
  }

}
