<?php

declare(strict_types=1);

namespace Drupal\panther\Drupal\Manager;

/**
 * Interface for classes that manage users created during tests.
 */
interface DrupalUserManagerInterface {

  /**
   * Returns the currently logged-in user.
   *
   * @return \stdClass|bool
   *   The user object, or FALSE if the user is anonymous.
   */
  public function getCurrentUser(): \stdClass|bool;

  /**
   * Sets the currently logged-in user.
   *
   * @param \stdClass|bool $user
   *   The user object, or FALSE if the user has been logged out.
   */
  public function setCurrentUser(\stdClass|bool $user): void;

  /**
   * Adds a new user.
   *
   * Call this after creating a new user to keep track of all the users that are
   * created in a test scenario. They can then be cleaned up after completing
   * the test.
   *
   * @param \stdClass $user
   *   The user object.
   *
   * @see \Drupal\DrupalExtension\Context\RawDrupalContext::cleanUsers()
   */
  public function addUser(\stdClass $user): void;

  /**
   * Removes a user from the list of users that were created in the test.
   *
   * @param string $userName
   *   The name of the user to remove.
   */
  public function removeUser(string $userName): void;

  /**
   * Returns the list of users that were created in the test.
   *
   * @return \stdClass[]
   *   An array of user objects.
   */
  public function getUsers(): array;

  /**
   * Returns the user with the given user name.
   *
   * @param string $userName
   *   The name of the user to return.
   *
   * @return \stdClass
   *   The user object.
   *
   * @throws \InvalidArgumentException
   *   Thrown when the user with the given name does not exist.
   */
  public function getUser(string $userName): \stdClass;

  /**
   * Clears the list of users that were created in the test.
   */
  public function clearUsers(): void;

  /**
   * Returns whether or not any users were created in the test.
   *
   * @return bool
   *   TRUE if any users are tracked, FALSE if not.
   */
  public function hasUsers(): bool;

  /**
   * Returns whether the current user is anonymous.
   *
   * @return bool
   *   TRUE if the current user is anonymous.
   */
  public function currentUserIsAnonymous(): bool;

}
