<?php

declare(strict_types=1);

namespace Drupal\panther\Drupal\Manager;

use Drupal\Core\Database\Transaction\TransactionManagerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Driver\DrupalDriver;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\media\Entity\Media;
use Drupal\media\MediaInterface;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\system\Entity\Menu;
use Drupal\system\MenuInterface;
use Drupal\user\UserStorage;

class EntityManager {

  /**
   * Keep track of nodes so they can be cleaned up.
   *
   * @var \stdClass[]
   */
  protected array $nodes = [];

  /**
   * Keep track of all terms that are created so they can easily be removed.
   *
   * @var \stdClass[]
   */
  protected array $terms = [];

  /**
   * Keep track of any roles that are created so they can easily be removed.
   *
   * @var string[]
   */
  protected array $roles = [];

  /**
   * Keep track of any languages that are created so they can easily be removed.
   *
   * @var \stdClass[]
   */
  protected array $languages = [];

  /**
   * Keep track of all menus that are created so they can easily be removed.
   *
   * @var \Drupal\system\MenuInterface[]
   */
  protected array $menus = [];

  /**
   * Keep track of all menu link that are created so they can easily be removed.
   *
   * @var \Drupal\menu_link_content\MenuLinkContentInterface[]
   */
  protected array $menuLinkContents = [];

  /**
   * Keep track of all media entities created so they can easily be removed.
   *
   * @var \Drupal\media\MediaInterface[]
   */
  protected array $medias = [];

  /**
   * Keep track of all file entities created so they can easily be removed.
   *
   * @var \Drupal\file\FileInterface[]
   */
  protected array $files = [];

  public function __construct(
    private readonly DrupalDriver $driver,
    private readonly DrupalUserManagerInterface $userManager,
    private readonly TransactionManagerBase $transactionManager,
    private readonly UserStorage $userStorage,
  ) {
  }

  /**
   * Create a node.
   *
   * @param \stdClass $node
   *   An object with, at least, the following properties:
   *   - title
   *   - type.
   *
   * @return object
   *   The created node.
   */
  public function nodeCreate(\stdClass $node): object {
    $this->parseEntityFields('node', $node);
    /** @var \stdClass $created */
    $created = $this->driver->createNode($node);
    $this->nodes[] = $created;

    return $created;
  }

  /**
   * Create a user.
   *
   * @param string[] $permissions
   *   An array of permissions to assign to the role.
   *   For example:
   *    - 'access administration pages'
   *    - 'access content'.
   *
   * @return string
   *   The created role name.
   */
  public function roleCreate(array $permissions): string {
    $role = $this->driver->roleCreate($permissions);
    $this->roles[] = (string) $role;

    return (string) $role;
  }

  /**
   * Create a user.
   *
   * @return object
   *   The created user.
   */
  public function userCreate(\stdClass $user): object {
    $this->parseEntityFields('user', $user);
    $this->driver->userCreate($user);
    $this->userManager->addUser($user);

    return $user;
  }

  /**
   * Create a term.
   *
   * @param \stdClass $term
   *   An object with, at least, the following properties:
   *    - name
   *    - type
   *    - vocabulary_machine_name
   *    - status.
   *
   * @return object
   *   The created term.
   */
  public function termCreate(\stdClass $term): object {
    $this->parseEntityFields('taxonomy_term', $term);
    /** @var \stdClass $created */
    $created = $this->driver->createTerm($term);
    $this->terms[] = $created;

    return $created;
  }

  /**
   * Creates a language.
   *
   * @param \stdClass $language
   *   An object with the following properties:
   *   - langcode: the langcode of the language to create.
   *
   * @return object|FALSE
   *   The created language, or FALSE if the language was already created.
   */
  public function languageCreate(\stdClass $language): object|false {
    /** @var \stdClass|false $created */
    $created = $this->driver->languageCreate($language);
    if ($created !== FALSE) {
      $this->languages[$created->langcode] = $created;
    }

    return $created;
  }

  /**
   * Create a menu.
   *
   * @return \Drupal\system\MenuInterface
   *   The created menu.
   */
  public function menuCreate(\stdClass $menu): MenuInterface {
    if (!\property_exists($menu, 'id')) {
      // Create menu id if one was not provided.
      $menu_id = \strtolower((string) $menu->label);
      $menu_id = \preg_replace('/[^a-z0-9_]+/', '_', $menu_id);
      $menu_id = \preg_replace('/_+/', '_', (string) $menu_id);
      $menu->id = $menu_id;
    }

    $created = Menu::create((array) $menu);
    $created->save();

    $this->menus[] = $created;

    return $created;
  }

  /**
   * Create a menu link content.
   *
   * @return \Drupal\menu_link_content\MenuLinkContentInterface
   *   The created menu link content.
   */
  public function menuLinkContentCreate(
    string $menu_name,
    \stdClass $menu_link_content,
  ): MenuLinkContentInterface {
    $menu = $this->loadMenuByLabel($menu_name);

    if (!$menu instanceof MenuInterface) {
      throw new \RuntimeException(\sprintf('Menu "%s" not found', $menu_name));
    }

    $menu_link_content->menu_name = $menu->id();
    // Add uri to correct property.
    $menu_link_content->link['uri'] = $menu_link_content->uri;
    unset($menu_link_content->uri);
    // Create parent property in format required.
    if (\property_exists($menu_link_content, 'parent')) {
      $parent_link = $this->loadMenuLinkByTitle(
        $menu_link_content->parent,
        $menu_name,
      );

      if ($parent_link !== NULL) {
        $menu_link_content->parent = 'menu_link_content:' . $parent_link->uuid();
      }
    }

    $created = MenuLinkContent::create((array) $menu_link_content);
    $created->save();
    $this->menuLinkContents[] = $created;

    return $created;
  }

  /**
   * Create a media entity.
   *
   * @param \stdClass $media
   *   An array with, at least, the following properties:
   *   [
   *   'name' => 'Name',
   *   'bundle' => 'media_bundle',
   *   'uid' => 1,
   *   'status' => 1,
   *   'field_media_file:target_id' => 1,
   *   ].
   *
   * @return \Drupal\media\MediaInterface
   *   The created media entity.
   */
  public function mediaCreate(\stdClass $media): MediaInterface {
    $this->parseEntityFields('media', $media);

    // Ensure bundle is set (required for media entities)
    if (!\property_exists($media, 'bundle')) {
      throw new \RuntimeException('Media bundle is required');
    }

    $created = Media::create((array) $media);
    $created->save();
    $this->medias[] = $created;

    return $created;
  }

  /**
   * Create a file entity.
   *
   * @param \stdClass $file
   *   An object with, at least, the following properties:
   *   - name
   *   - uri (file path).
   *
   * @return \Drupal\file\FileInterface
   *   The created file entity.
   */
  public function fileCreate(\stdClass $file): FileInterface {
    $this->parseEntityFields('file', $file);

    // Ensure uri is set (required for file entities)
    if (!\property_exists($file, 'uri')) {
      throw new \RuntimeException('File uri is required');
    }

    $created = File::create((array) $file);
    $created->save();
    $this->files[] = $created;

    return $created;
  }

  public function addUserToRoles(\stdClass $user, string $roles): void {
    $roles = \explode(',', $roles);
    $roles = \array_map('trim', $roles);
    foreach ($roles as $role) {
      if (!\in_array(\strtolower($role), [
        'authenticated',
        'authenticated user',
      ], TRUE)) {
        // Only add roles other than 'authenticated user'.
        $this->driver->userAddRole($user, $role);
      }
    }
  }

  /**
   * Remove any created nodes.
   */
  public function cleanNodes(): void {
    // Remove any nodes that were created.
    foreach ($this->nodes as $node) {
      $this->driver->nodeDelete($node);
    }
    $this->nodes = [];
  }

  /**
   * Remove any created roles.
   */
  public function cleanRoles(): void {
    // Remove any roles that were created.
    foreach ($this->roles as $rid) {
      $this->driver->roleDelete($rid);
    }
    $this->roles = [];
  }

  /**
   * Remove any created users.
   */
  public function cleanUsers(): void {
    // Remove any users that were created.
    if ($this->userManager->hasUsers()) {
      foreach ($this->userManager->getUsers() as $user) {
        $this->driver->userDelete($user);
      }
      $this->driver->processBatch();
      $this->userManager->clearUsers();
    }
  }

  /**
   * Remove any created terms.
   */
  public function cleanTerms(): void {
    // Remove any terms that were created.
    foreach ($this->terms as $term) {
      $this->driver->termDelete($term);
    }
    $this->terms = [];
  }

  /**
   * Remove any created languages.
   */
  public function cleanLanguages(): void {
    // Remove any languages that were created.
    foreach ($this->languages as $language) {
      $this->driver->languageDelete($language);
      unset($this->languages[$language->langcode]);
    }
  }

  /**
   * Remove any created menu.
   */
  public function cleanMenu(): void {
    // Remove any menu that were created.
    foreach ($this->menus as $menu) {
      if ($menu instanceof EntityInterface) {
        $menu->delete();
      }
    }
    $this->menus = [];
  }

  /**
   * Remove any created menu link contents.
   */
  public function cleanMenuLinkContents(): void {
    // Remove any menu link contents that were created.
    foreach ($this->menuLinkContents as $menu_link_content) {
      if ($menu_link_content instanceof EntityInterface) {
        $menu_link_content->delete();
      }
    }
    $this->menuLinkContents = [];
  }

  /**
   * Remove any created media.
   */
  public function cleanMedias(): void {
    // Remove any media that were created.
    foreach ($this->medias as $media) {
      if ($media instanceof EntityInterface) {
        $media->delete();
      }
    }
    $this->medias = [];
  }

  /**
   * Remove any created files.
   */
  public function cleanFiles(): void {
    // Remove any files that were created.
    foreach ($this->files as $file) {
      if ($file instanceof EntityInterface) {
        $file->delete();
      }
    }
    $this->files = [];
  }

  public function cleanAll(): void {
    $this->cleanNodes();
    $this->cleanUsers();
    $this->cleanTerms();
    $this->cleanRoles();
    $this->cleanLanguages();
    $this->cleanMenu();
    $this->cleanMenuLinkContents();
    $this->cleanMedias();
    $this->cleanFiles();

    // Commit all transactions to ensure that the database is in a clean state.
    $this->transactionManager->commitAll();
  }

  /**
   * Delete a user by email.
   */
  public function deleteUserByEmail(string $email): void {
    /** @var \Drupal\user\UserInterface[] $users */
    $users = $this->userStorage->loadByProperties(['mail' => $email]);

    if (\count($users) > 0) {
      $user = \reset($users);
      $this->driver->userDelete((object) ['uid' => $user->id()]);
      $this->driver->processBatch();
    }
  }

  /**
   * Delete a file entity by URI.
   */
  public function deleteFileByUri(string $uri): void {
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager */
    // @phpstan-ignore-next-line
    $entity_type_manager = \Drupal::getContainer()->get('entity_type.manager');

    /** @var \Drupal\file\FileInterface[] $files */
    $files = $entity_type_manager->getStorage('file')
      ->loadByProperties(['uri' => $uri]);

    if (\count($files) > 0) {
      foreach ($files as $file) {
        $file->delete();
      }
    }
  }

  /**
   * Parses the field values and turns them into the format expected by Drupal.
   *
   * Multiple values in a single field must be separated by commas. Wrap the
   * field value in double quotes in case it should contain a comma.
   *
   * Compound field properties are identified using a ':' operator, either in
   * the column heading or in the cell. If multiple properties are present in a
   * single cell, they must be separated using ' - ', and values should not
   * contain ':' or ' - '.
   *
   * Possible formats for the values:
   *   A
   *   A, B, "a value, containing a comma"
   *   A - B
   *   x: A - y: B
   *   A - B, C - D, "E - F"
   *   x: A - y: B,  x: C - y: D,  "x: E - y: F"
   *
   * See field_handlers.feature for examples of usage.
   *
   * @param string $entity_type
   *   The entity type.
   * @param object $entity
   *   An object containing the entity properties and fields as properties.
   *
   * @throws \Exception
   *   Thrown when a field name is invalid.
   */
  private function parseEntityFields(string $entity_type, object $entity): void {
    $multicolumn_field = '';
    $multicolumn_column = '';
    $multicolumn_fields = [];

    // Convert the entity to an array to iterate over the properties.
    /** @var string[] $entity_properties */
    $entity_properties = \json_decode((string) \json_encode($entity), TRUE);
    foreach ($entity_properties as $field => $field_value) {
      // Reset the multicolumn field if the field name does not contain a
      // column.
      if (!\str_contains($field, ':')) {
        $multicolumn_field = '';
      }
      elseif (\strpos($field, ':', 1) !== FALSE) {
        // Start tracking a new multicolumn field if the field name contains a
        // ':' which is preceded by at least 1 character.
        [$multicolumn_field, $multicolumn_column] = \explode(':', $field);
      }
      elseif ($multicolumn_field === '') {
        // If a field name starts with a ':' but we are not yet tracking a
        // multicolumn field we don't know to which field this belongs.
        throw new \Exception('Field name missing for ' . $field);
      }
      else {
        // Update the column name if the field name starts with a ':' and we are
        // already tracking a multicolumn field.
        $multicolumn_column = \substr($field, 1);
      }

      $is_multicolumn = $multicolumn_field !== '' && $multicolumn_column !== '';
      $field_name = $multicolumn_field !== '' ? $multicolumn_field : $field;
      if ($this->driver->isField($entity_type, $field_name)) {
        // Split up multiple values in multi-value fields.
        $values = [];
        foreach (\str_getcsv($field_value) as $key => $value) {
          $value = \trim((string) $value);
          $columns = $value;
          // Split up field columns if the ' - ' separator is present.
          if (\str_contains($value, ' - ')) {
            $columns = [];
            foreach (\explode(' - ', $value) as $column) {
              // Check if it is an inline named column.
              if (!$is_multicolumn && \strpos($column, ': ', 1) !== FALSE) {
                [$key, $column] = \explode(': ', $column);
                $columns[$key] = $column;
              }
              else {
                $columns[] = $column;
              }
            }
          }
          // Use the column name if we are tracking a multicolumn field.
          if ($is_multicolumn) {
            $multicolumn_fields[$multicolumn_field][$key][$multicolumn_column] = $columns;
            // @phpstan-ignore-next-line
            unset($entity->$field);
          }
          else {
            $values[] = $columns;
          }
        }
        // Replace regular fields inline in the entity after parsing.
        if (!$is_multicolumn) {
          // @phpstan-ignore-next-line
          $entity->$field_name = $values;
          // Don't specify any value if the step author has left it blank.
          if ($field_value === '') {
            // @phpstan-ignore-next-line
            unset($entity->$field_name);
          }
        }
      }
    }

    // Add the multicolumn fields to the entity.
    foreach ($multicolumn_fields as $field_name => $columns) {
      // Don't specify any value if the step author has left it blank.
      if (\count(\array_filter($columns, static function ($var) {
          // @phpstan-ignore-next-line
          return ($var !== '');
      })) > 0) {
        // @phpstan-ignore-next-line
        $entity->$field_name = $columns;
      }
    }
  }

  /**
   * Get a menu by label.
   *
   * @param string $label
   *   The label of the menu.
   *
   * @return \Drupal\system\MenuInterface|null
   *   The menu or NULL if not found.
   */
  private function loadMenuByLabel(string $label): ?MenuInterface {
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager */
    // @phpstan-ignore-next-line
    $entity_type_manager = \Drupal::getContainer()->get('entity_type.manager');
    $menu_ids = $entity_type_manager->getStorage('menu')->getQuery()
      ->accessCheck(FALSE)
      ->condition('label', $label)
      ->execute();

    if (\count($menu_ids) === 0) {
      return NULL;
    }

    $menu_id = \reset($menu_ids);

    return Menu::load($menu_id);
  }

  /**
   * Get a menu link by title and menu name.
   *
   * @param string $title
   *   The title of the menu link.
   * @param string $menu_name
   *   The name of the menu.
   *
   * @return \Drupal\menu_link_content\Entity\MenuLinkContent|null
   *   The menu link or NULL if not found.
   */
  private function loadMenuLinkByTitle(string $title, string $menu_name): ?MenuLinkContent {
    $menu = $this->loadMenuByLabel($menu_name);

    if (!$menu instanceof MenuInterface) {
      return NULL;
    }

    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager */
    // @phpstan-ignore-next-line
    $entity_type_manager = \Drupal::getContainer()->get('entity_type.manager');

    $menu_link_ids = $entity_type_manager->getStorage('menu_link_content')->getQuery()
      ->accessCheck(FALSE)
      ->condition('menu_name', $menu->id())
      ->condition('title', $title)
      ->execute();

    if (\count($menu_link_ids) === 0) {
      return NULL;
    }

    $menu_link_id = \reset($menu_link_ids);

    return MenuLinkContent::load($menu_link_id);
  }

}
