<?php

declare(strict_types=1);

namespace Drupal\panther;

use Symfony\Component\OptionsResolver\OptionsResolver;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Validation;

class Settings {


  public readonly string $screenshotsDir;

  public readonly string $accessibilityReportsDir;

  public readonly string $loginUrl;

  public readonly string $logoutUrl;

  public readonly string $loginButton;

  public readonly string $logoutButton;

  public readonly string $usernameField;

  public readonly string $passwordField;

  public readonly string $axeScriptUrl;

  public readonly bool $failOnAxeError;

  public readonly bool $dumpAccessibilityReports;

  /**
   * @var string[]
   */
  public readonly array $axeTags;

  public readonly bool $axeBestPractices;

  public function __construct() {
    // @phpstan-ignore-next-line
    $container = \Drupal::getContainer();
    $parameters = [];
    if ($container->hasParameter('panther')) {
      /** @var string[] $parameters */
      $parameters = $container->getParameter('panther');
    }
    $resolver = new OptionsResolver();

    $not_blank = Validation::createIsValidCallable(
      new NotBlank(),
    );

    $resolver
      ->define('screenshots_dir')
      ->allowedTypes('string')
      ->default('/var/www/html/web/sites/default/files/panther_screenshots')
      ->allowedValues($not_blank);

    $resolver
      ->define('accessibility_reports_dir')
      ->allowedTypes('string')
      ->default('/var/www/html/web/sites/default/files/panther_accessibility_reports')
      ->allowedValues($not_blank);

    $resolver
      ->define('login_url')
      ->allowedTypes('string')
      ->default('/user/login')
      ->allowedValues($not_blank);

    $resolver
      ->define('logout_url')
      ->allowedTypes('string')
      ->default('/user/logout')
      ->allowedValues($not_blank);

    $resolver
      ->define('login_button')
      ->allowedTypes('string')
      ->default('Log in')
      ->allowedValues($not_blank);

    $resolver
      ->define('logout_button')
      ->allowedTypes('string')
      ->default('Log out')
      ->allowedValues($not_blank);

    $resolver
      ->define('username_field')
      ->allowedTypes('string')
      ->default('name')
      ->allowedValues($not_blank);

    $resolver
      ->define('password_field')
      ->allowedTypes('string')
      ->default('pass')
      ->allowedValues($not_blank);

    $resolver
      ->define('axe_script_url')
      ->allowedTypes('string')
      ->default(
        'https://cdnjs.cloudflare.com/ajax/libs/axe-core/4.10.3/axe.min.js',
      )
      ->allowedValues($not_blank);

    $resolver
      ->define('axe_tags')
      ->allowedTypes('array')
      ->default(['wcag2a', 'wcag2aa', 'wcag21a', 'wcag21aa']);

    $resolver
      ->define('axe_best_practices')
      ->allowedTypes('boolean')
      ->default(TRUE);

    $resolver
      ->define('fail_on_axe_error')
      ->allowedTypes('boolean')
      ->default(FALSE);

    $resolver
      ->define('dump_accessibility_reports')
      ->allowedTypes('boolean')
      ->default(TRUE);

    $values = $resolver->resolve($parameters);

    $this->screenshotsDir = $values['screenshots_dir'];
    $this->accessibilityReportsDir = $values['accessibility_reports_dir'];
    $this->loginUrl = $values['login_url'];
    $this->logoutUrl = $values['logout_url'];
    $this->loginButton = $values['login_button'];
    $this->logoutButton = $values['logout_button'];
    $this->usernameField = $values['username_field'];
    $this->passwordField = $values['password_field'];
    $this->axeScriptUrl = $values['axe_script_url'];
    $this->axeTags = $values['axe_tags'];
    $this->axeBestPractices = $values['axe_best_practices'];
    $this->failOnAxeError = $values['fail_on_axe_error'];
    $this->dumpAccessibilityReports = $values['dump_accessibility_reports'];
  }

}
