<?php

declare(strict_types=1);

namespace Drupal\Tests\panther\FunctionalJavascript;

use PHPUnit\Framework\Constraint\LogicalAnd;
use PHPUnit\Framework\Constraint\LogicalNot;
use PHPUnit\Framework\Constraint\StringContains;
use Symfony\Component\DomCrawler\Crawler;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerAnySelectorTextContains;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerAnySelectorTextSame;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerSelectorAttributeValueSame;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerSelectorCount;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerSelectorExists;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerSelectorTextContains;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerSelectorTextSame;

trait DomCrawlerAssertionsTrait {

  public static function assertPageContains(
    string $string,
    string $message = '',
  ): void {
    self::assertThat(
      self::getClient()->getPageSource(),
      new StringContains($string),
      $message === ''
        ? \sprintf('The page does not contain the string "%s".', $string)
        : $message,
    );
  }

  public static function assertPageNotContains(
    string $string,
    string $message = '',
  ): void {
    self::assertThat(
      self::getClient()->getPageSource(),
      new LogicalNot(new StringContains($string)),
      $message === ''
        ? \sprintf('The page does contain the string "%s".', $string)
        : $message,);
  }

  public static function assertSelectorExists(
    string $selector,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new CrawlerSelectorExists($selector),
      $message === ''
        ? \sprintf('The selector "%s" does not exist.', $selector)
        : $message,
    );
  }

  public static function assertSelectorNotExists(
    string $selector,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new LogicalNot(new CrawlerSelectorExists($selector)),
      $message === ''
        ? \sprintf('The selector "%s" exists.', $selector)
        : $message,);
  }

  public static function assertSelectorCount(
    int $expectedCount,
    string $selector,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new CrawlerSelectorCount($expectedCount, $selector),
      $message === ''
        ? \sprintf('The selector "%s" does not exists %d times.', $selector, $expectedCount)
        : $message,);
  }

  public static function assertSelectorTextContains(
    string $selector,
    string $text,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists($selector),
        new CrawlerSelectorTextContains($selector, $text),
      ),
      $message === ''
        ? \sprintf('The selector "%s" does not contain the text "%s".', $selector, $text)
        : $message,);
  }

  public static function assertAnySelectorTextContains(
    string $selector,
    string $text,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists($selector),
        new CrawlerAnySelectorTextContains(
          $selector,
          $text,
        ),
      ),
      $message === ''
        ? \sprintf('The selector "%s" does not contain the text "%s".', $selector, $text)
        : $message,);
  }

  public static function assertSelectorTextSame(
    string $selector,
    string $text,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists($selector),
        new CrawlerSelectorTextSame($selector, $text),
      ),
      $message === ''
        ? \sprintf('The selector "%s" text is not the same as "%s".', $selector, $text)
        : $message,);
  }

  public static function assertAnySelectorTextSame(
    string $selector,
    string $text,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists($selector),
        new CrawlerAnySelectorTextSame($selector, $text),
      ),
      $message === ''
        ? \sprintf('The selector "%s" text is not the same as "%s".', $selector, $text)
        : $message,);
  }

  public static function assertSelectorTextNotContains(
    string $selector,
    string $text,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists($selector),
        new LogicalNot(
          new CrawlerSelectorTextContains($selector, $text),
        ),
      ),
      $message === ''
        ? \sprintf('The selector "%s" contains the text "%s".', $selector, $text)
        : $message,);
  }

  public static function assertAnySelectorTextNotContains(
    string $selector,
    string $text,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists($selector),
        new LogicalNot(
          new CrawlerAnySelectorTextContains(
            $selector,
            $text,
          ),
        ),
      ),
      $message === ''
        ? \sprintf('The selector "%s" contains the text "%s".', $selector, $text)
        : $message,
    );
  }

  public static function assertPageTitleSame(
    string $expectedTitle,
    string $message = '',
  ): void {
    self::assertSelectorTextSame(
      'title',
      $expectedTitle,
      $message === ''
        ? \sprintf('The page title is not the same as "%s".', $expectedTitle)
        : $message,
    );
  }

  public static function assertPageTitleContains(
    string $expectedTitle,
    string $message = '',
  ): void {
    self::assertSelectorTextContains(
      'title',
      $expectedTitle,
      $message === ''
        ? \sprintf('The page title does not contain the text "%s".', $expectedTitle)
        : $message,
    );
  }

  public static function assertInputValueSame(
    string $fieldName,
    string $expectedValue,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists("input[name=\"$fieldName\"]"),
        new CrawlerSelectorAttributeValueSame(
          "input[name=\"$fieldName\"]", 'value', $expectedValue,
        ),
      ),
      $message === ''
        ? \sprintf('The input "%s" does not exist or has a different value.', $fieldName)
        : $message,
    );
  }

  public static function assertInputValueNotSame(
    string $fieldName,
    string $expectedValue,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      LogicalAnd::fromConstraints(
        new CrawlerSelectorExists("input[name=\"$fieldName\"]"),
        new LogicalNot(
          new CrawlerSelectorAttributeValueSame(
            "input[name=\"$fieldName\"]", 'value', $expectedValue,
          ),
        ),
      ),
      $message === ''
        ? \sprintf('The input "%s" does not exist or has the same value.', $fieldName)
        : $message,
    );
  }

  public static function assertCheckboxChecked(
    string $fieldName,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new CrawlerSelectorExists("input[name=\"$fieldName\"]:checked"),
      $message === ''
        ? \sprintf('The checkbox "%s" is not checked.', $fieldName)
        : $message,
    );
  }

  public static function assertCheckboxNotChecked(
    string $fieldName,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new LogicalNot(
        new CrawlerSelectorExists("input[name=\"$fieldName\"]:checked"),
      ),
      $message === ''
        ? \sprintf('The checkbox "%s" is checked.', $fieldName)
        : $message,
    );
  }

  public static function assertFormValue(
    string $formSelector,
    string $fieldName,
    string $value,
    string $message = '',
  ): void {
    $node = self::getCrawler()->filter($formSelector);
    self::assertNotEmpty(
      $node,
      \sprintf('Form "%s" not found.', $formSelector),
    );
    $values = $node->form()->getValues();
    self::assertArrayHasKey(
      $fieldName,
      $values,
      $message !== '' ? $message : \sprintf(
        'Field "%s" not found in form "%s".',
        $fieldName,
        $formSelector,
      ),
    );
    self::assertSame($value, $values[$fieldName]);
  }

  public static function assertNoFormValue(
    string $formSelector,
    string $fieldName,
    string $message = '',
  ): void {
    $node = self::getCrawler()->filter($formSelector);
    self::assertNotEmpty(
      $node,
      \sprintf('Form "%s" not found.', $formSelector),
    );
    $values = $node->form()->getValues();
    self::assertArrayNotHasKey(
      $fieldName,
      $values,
      $message !== '' ? $message : \sprintf(
        'Field "%s" has a value in form "%s".',
        $fieldName,
        $formSelector,
      ),
    );
  }

  /**
   * @phpstan-param string[] $expected_values
   */
  public static function assertEach(
    callable $callback,
    string $selector,
    array $expected_values,
    string $message = '',
  ): void {
    $crawler = self::getCrawler();
    $elements = $crawler->filter($selector);

    self::assertCount(
      \count($elements),
      $expected_values,
      'The number of elements does not match the number of expected values.',
    );

    $elements->each(
      static function (Crawler $element, $index) use (
        $expected_values,
        $callback,
        $message
      ): void {
        $callback($element, $expected_values[$index], $message);
      },
    );
  }

  private static function getCrawler(): Crawler {
    try {
      return self::getClient()->getCrawler();
    }
    catch (\BadMethodCallException $e) {
      static::fail(
        'A client must have a crawler to make assertions. Did you forget to make an HTTP request?',
      );
    }
  }

}
