<?php

declare(strict_types=1);

namespace Drupal\Tests\panther\FunctionalJavascript;

use Drupal\Tests\panther\FunctionalJavascript\Constraint\CrawlerSelectorDisabled;
use Drupal\Tests\panther\FunctionalJavascript\Constraint\CrawlerSelectorEnabled;
use Facebook\WebDriver\Exception\NoSuchElementException;
use Facebook\WebDriver\Exception\TimeoutException;
use Facebook\WebDriver\WebDriverBy;
use PHPUnit\Framework\Constraint\LogicalNot;
use Symfony\Component\DomCrawler\Crawler;
use Symfony\Component\DomCrawler\Test\Constraint\CrawlerSelectorTextContains;
use Symfony\Component\Panther\DomCrawler\Crawler as PantherCrawler;

/**
 * Provides assertions for Drupal.
 */
trait DrupalAssertionsTrait {

  /**
   * Asserts that a message contains a header and a message.
   *
   * @param string $header
   *   The header.
   * @param string $message
   *   The message.
   */
  public function assertMessageContains(string $header, string $message): void {
    try {
      self::getClient()->waitFor('.messages-list');
      self::assertPageContains($header);
      self::assertPageContains($message);
    }
    catch (NoSuchElementException | TimeoutException $e) {
      self::fail('No message found.');
    }
  }

  /**
   * Assert that a page contains a text a specific number of times.
   *
   * @param string $text
   *   The text to search for.
   * @param int $times
   *   The number of times the text should appear.
   */
  public function assertPageContainsNumTimes(string $text, int $times): void {
    self::assertEquals(
      $times,
      self::getCrawler()
        ->filterXPath('//*[normalize-space(text())="' . $text . '"]')
        ->count(),
    );
  }

  /**
   * Assert that a table row contains a specific text.
   *
   * @param string $text
   *   The text to locate the row.
   * @param int $column
   *   The column to look for the text.
   * @param string $value
   *   The value to search for.
   */
  public function assertTableRowWithTextContains(string $text, int $column, string $value): void {
    $row = \sprintf(
      '//table//tr[td[%d][contains(., \'%s\')]]',
      $column,
      $text,
    );

    $success = FALSE;
    self::getCrawler()->filterXPath($row)->each(
      static function ($node) use ($value, &$success): void {
        self::assertStringContainsString($value, $node->text());
        $success = TRUE;
      },
    );

    if (!$success) {
      self::fail(\sprintf('No row found with text "%s" in column %d.', $text, $column));
    }
  }

  public static function assertElementDisabled(
    string $selector,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new CrawlerSelectorDisabled($selector),
      $message,
    );
  }

  public static function assertElementEnabled(
    string $selector,
    string $message = '',
  ): void {
    self::assertThat(
      self::getCrawler(),
      new CrawlerSelectorEnabled($selector),
      $message,
    );
  }

  public static function assertLoggedIn(): void {
    self::assertPageContains('Log out');
  }

  public static function assertResponseStatusCodeSame(int $status_code, string $message = ''): void {
    try {
      $response_status = self::getClient()
        ->executeScript('return window.performance.getEntries()[0].responseStatus');
    }
    catch (\Exception $e) {
      $response_status = 0;
    }

    self::assertEquals(
      $status_code,
      $response_status,
      $message,
    );
  }

  public static function assertUrlMatches(string $url): void {
    // Convert the URL to a regex pattern.
    $url = \str_replace('/', '\/', $url);
    $url = '/.*' . $url . '/';

    // Get the current URL.
    $current_url = self::getClient()->getCurrentURL();

    self::assertTrue(
      \preg_match($url, $current_url) === 1,
      \sprintf(
        'The URL "%s" does not match the expected pattern "%s".',
        $current_url,
        $url,
      ),
    );
  }

  /**
   * Assert that a select element contains specific options.
   *
   * @param string $selector
   *   The CSS selector to filter the DOM.
   * @param array<string, string> $options
   *   An associative array of options where the key is the option value.
   * @param string $message
   *   A custom message to display if the assertion fails.
   */
  public static function assertSelectContainsOptions(
    string $selector,
    array $options,
    string $message = '',
  ): void {
    $crawler = self::getCrawler();
    $node = $crawler->filter($selector);
    self::assertNotEmpty(
      $node,
      \sprintf('The selector "%s" does not exist.', $selector),
    );

    foreach ($options as $option_value => $option_label) {
      self::assertThat(
        $node,
        new CrawlerSelectorTextContains(
          "option[value=\"$option_value\"]",
          $option_label,
        ),
        $message === ''
          ? \sprintf(
          'The select "%s" does not contain the option "%s".',
          $selector,
          $option_label,
        )
          : $message,
      );
    }
  }

  /**
   * Assert that a select element does not contain specific options.
   *
   * @param string $selector
   *   The CSS selector to filter the DOM.
   * @param array<string, string> $options
   *   An associative array of options where the key is the option value.
   * @param string $message
   *   A custom message to display if the assertion fails.
   */
  public static function assertSelectNotContainsOptions(
    string $selector,
    array $options,
    string $message = '',
  ): void {
    $crawler = self::getCrawler();
    $node = $crawler->filter($selector);
    self::assertNotEmpty(
      $node,
      \sprintf('The selector "%s" does not exist.', $selector),
    );
    foreach ($options as $option_value => $option_label) {
      self::assertThat(
        $node,
        new LogicalNot(
          new CrawlerSelectorTextContains(
            "option[value=\"$option_value\"]",
            $option_label,
          ),
        ),
        $message === ''
          ? \sprintf(
          'The select "%s" contains the option "%s".',
          $selector,
          $option_label,
        )
          : $message,
      );
    }
  }

  /**
   * Assert that a table row does not have a specific operation.
   *
   * @param string $text
   *   The text to search for in the specified column.
   * @param int $column
   *   The column index (1-based) to search for the text.
   * @param string $operation_value
   *   The operation value to search for.
   * @param string|null $message
   *   A custom message to display if the assertion fails.
   */
  public function assertOperationsNotContainOnTableRowWithText(
    string $text,
    int $column,
    string $operation_value,
    ?string $message = NULL,
  ): void {
    $message ??= \sprintf('The operations contains the string %s', $operation_value);
    $operation_elements = $this->filterTableRowOperationElements($text, $column);

    $operation_elements->each(static function ($element) use ($operation_value, $message): void {
      self::assertStringNotContainsString(
        $operation_value,
        $element->getText(),
        $message,
      );
    });

    // Reload the page to reset the crawler state.
    self::getClient()->reload();
  }

  /**
   * Assert that a table row containing specific text has a specific operation.
   *
   * @param string $text
   *   The text to search for in the specified column.
   * @param int $column
   *   The column index (1-based) to search for the text.
   * @param string $operation_value
   *   The operation value to search for.
   * @param string|null $message
   *   A custom message to display if the assertion fails.
   */
  public function assertOperationsContainOnTableRowWithText(
    string $text,
    int $column,
    string $operation_value,
    ?string $message = NULL,
  ): void {
    $message ??= \sprintf('The operations contains the string %s', $operation_value);
    $operation_elements = $this->filterTableRowOperationElements($text, $column);

    if ($message === '') {
      $message = \sprintf('Expected at least one row to contain operation "%s"', $operation_value);
    }

    $operation_found = FALSE;

    $operation_elements->each(static function ($element) use ($operation_value, &$operation_found): void {
      if (\str_contains($element->getText(), $operation_value)) {
        $operation_found = TRUE;
      }
    });

    self::assertTrue($operation_found, $message);

    // Reload the page to reset the crawler state.
    self::getClient()->reload();
  }

  /**
   * Select operation element(s) on a table row containing specific text.
   */
  private function filterTableRowOperationElements(string $text, int $column): Crawler {
    $row_value = \sprintf(
      '//table//tr[td[%d][contains(., \'%s\')]]',
      $column,
      $text,
    );

    $operations_selector = \sprintf(
      '//table//tr[td[%d][contains(text(), \'%s\')]]//div[@class="dropbutton-widget"]//li',
      $column,
      $text,
    );

    $crawler = self::getCrawler();

    $this->scrollDownUntilVisible(WebDriverBy::xpath($row_value));
    $row = $crawler->filterXPath($row_value);
    $dropdown_toggle = $row->filter('.dropbutton__toggle');

    \assert($dropdown_toggle instanceof PantherCrawler);

    if ($dropdown_toggle->count() > 0) {
      // The operation links are more than one, so it's necessary to click
      // the dropdown button to show them.
      $dropdown_toggle->click();

      try {
        self::getClient()->waitFor('.dropbutton__items');
      }
      catch (\Exception $e) {
        self::fail($e->getMessage());
      }
    }

    $operation_elements = $crawler->filterXPath($operations_selector);

    self::assertNotEmpty(
      $operation_elements,
      \sprintf('The selector "%s" does not exist.', $operations_selector),
    );

    return $operation_elements;
  }

}
