<?php

declare(strict_types=1);

namespace Drupal\Tests\panther\FunctionalJavascript;

use Drupal\Core\File\FileExists;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;
use Drupal\media\MediaInterface;
use Drupal\menu_link_content\MenuLinkContentInterface;
use Drupal\system\MenuInterface;
use Drupal\user\UserInterface;
use Facebook\WebDriver\Exception\NoSuchElementException;
use Facebook\WebDriver\Exception\StaleElementReferenceException;
use Facebook\WebDriver\WebDriver;
use Facebook\WebDriver\WebDriverBy;
use Facebook\WebDriver\WebDriverElement;
use Facebook\WebDriver\WebDriverExpectedCondition;
use Facebook\WebDriver\WebDriverKeys;

/**
 * Provides Drupal-related methods.
 */
trait DrupalTrait {

  /**
   * Logs in as a user.
   *
   * @param string $user
   *   The user's name.
   * @param string $password
   *   The password.
   */
  public function loginAs(string $user, string $password): void {
    $login_url = $this->settings->loginUrl;
    $login_button = $this->settings->loginButton;
    $username_field = $this->settings->usernameField;
    $password_field = $this->settings->passwordField;

    $this->goToPage($login_url);
    $this->maximizeWindow();
    self::assertPageContains($login_button);

    $this->submitForm($login_button, [
      $username_field => $user,
      $password_field => $password,
    ]);
  }

  /**
   * Logout from the current user.
   */
  public function logout(): void {
    $logout_url = $this->settings->logoutUrl;
    $logout_button = $this->settings->logoutButton;

    $this->goToPage($logout_url);
    $this->maximizeWindow();
    self::assertPageContains($logout_button);
    $this->submitForm($logout_button, []);
  }

  /**
   * Select an operation on a table row with a given text.
   *
   * @param string $text
   *   The text to locate the row.
   * @param int $column
   *   The column to look for the text.
   * @param string $operation
   *   The operation to perform.
   */
  public function selectOperationOnTableRowWithText(
    string $text,
    int $column,
    string $operation,
  ): void {
    $value = \sprintf(
      '//table//tr[td[%d][contains(., \'%s\')]]',
      $column,
      $text,
    );

    $this->scrollDownUntilVisible(WebDriverBy::xpath($value));

    $crawler = self::getClient()->getCrawler();

    $row = $crawler->filterXPath($value);
    $dropdown_toggle = $row->filter('.dropbutton__toggle');

    try {
      // The operation links are more than one, so it's necessary to click
      // the dropdown button to show them.
      if ($dropdown_toggle->count() > 0) {
        $dropdown_toggle->click();
        self::getClient()->waitFor('.dropbutton__items');
      }

      $row->selectLink($operation)->click();

      // Reload the page to reset the crawler state.
      self::getClient()->reload();
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

  /**
   * Select an option from a select element whose label is given.
   *
   * @param string $label
   *   The label of the select element.
   * @param string $value
   *   The value of the option to select.
   */
  public function selectOption(string $label, string $value): void {
    $crawler = self::getClient()->getCrawler();

    $xpath = \sprintf(
      ".//label[text()[contains(.,'%s')]][1]/following-sibling::select[1]//option[@value='%s']",
      $label,
      $value,
    );
    $option = $crawler->filterXPath($xpath);
    $option->click();
  }

  /**
   * Scroll down until the given element is visible.
   *
   * @param \Facebook\WebDriver\WebDriverBy $by
   *   The WebDriverBy object.
   */
  public function scrollDownUntilVisible(WebDriverBy $by): void {
    try {
      $element = self::getClient()->findElement($by);
      while (!$element->isDisplayed()) {
        self::getClient()->getKeyboard()->pressKey(WebDriverKeys::PAGE_DOWN);
      }
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

  /**
   * Scroll down until the given element is visible.
   *
   * @param string $selector
   *   The CSS selector.
   */
  public function scrollDownUntilVisibleBySelector(string $selector): void {
    $this->scrollDownUntilVisible(WebDriverBy::cssSelector($selector));
  }

  public function clickSubmitButton(): void {
    $this->waitForElementToBeClickable(".//input[@id='edit-submit']")->click();
  }

  /**
   * Create a node.
   *
   * @phpstan-param array<string, mixed> $node
   */
  public function nodeCreate(
    array $node,
  ): \stdClass {
    /** @var \stdClass $created */
    $created = $this->entityManager->nodeCreate((object) $node);

    return $created;
  }

  /**
   * Create a user.
   *
   * @phpstan-param array<string, mixed> $user
   */
  public function userCreate(
    array $user,
  ): \stdClass {
    $user = (object) $user;

    if (!isset($user->name)) {
      $user->name = $this->driver->getRandom()->name(8);
    }
    if (!isset($user->pass)) {
      $user->pass = $this->driver->getRandom()->name(8);
    }
    if (!isset($user->mail)) {
      $user->mail = "{$user->name}@example.com";
    }

    /** @var \stdClass $created */
    $created = $this->entityManager->userCreate($user);

    return $created;
  }

  /**
   * Create a role.
   *
   * @phpstan-param string[] $permissions
   */
  public function roleCreate(
    array $permissions,
  ): string {
    return $this->entityManager->roleCreate($permissions);
  }

  /**
   * Create a term.
   *
   * @phpstan-param array<string, mixed> $term
   */
  public function termCreate(
    array $term,
  ): \stdClass {
    /** @var \stdClass $created */
    $created = $this->entityManager->termCreate((object) $term);

    return $created;
  }

  /**
   * Create a language.
   *
   * @phpstan-param array<string, mixed> $language
   */
  public function languageCreate(
    array $language,
  ): \stdClass {
    /** @var \stdClass $created */
    $created = $this->entityManager->languageCreate((object) $language);

    return $created;
  }

  /**
   * Create a menu.
   *
   * @phpstan-param array<string, mixed> $menu
   */
  public function menuCreate(
    array $menu,
  ): MenuInterface {
    return $this->entityManager->menuCreate((object) $menu);
  }

  /**
   * Create a menu link content.
   *
   * @phpstan-param array<string, mixed> $menu_link_content
   */
  public function menuLinkContentCreate(
    string $menu_name,
    array $menu_link_content,
  ): MenuLinkContentInterface {
    return $this
      ->entityManager
      ->menuLinkContentCreate(
        $menu_name,
        (object) $menu_link_content,
      );
  }

  /**
   * Create a media entity.
   *
   * @phpstan-param array<string, mixed> $media
   */
  public function mediaCreate(
    array $media,
  ): MediaInterface {
    return $this->entityManager->mediaCreate((object) $media);
  }

  /**
   * Create a file entity.
   *
   * @phpstan-param array<string, mixed> $file
   */
  public function fileCreate(
    array $file,
    string $filepath,
    string $directory = 'public://',
  ): FileInterface {
    /** @var \Drupal\Core\File\FileSystem $file_system */
    $file_system = \Drupal::service('file_system');
    $file_system->prepareDirectory($directory, FileSystemInterface:: CREATE_DIRECTORY);
    $file_system->copy($filepath, $directory . '/' . \basename($filepath), FileExists::Rename);

    $file += [
      'filename' => \basename($filepath),
      'uri' => $directory . '/' . \basename($filepath),
    ];

    return $this->entityManager->fileCreate((object) $file);
  }

  /**
   * Delete a user by email.
   */
  public function deleteUserByEmail(string $email): void {
    try {
      $this->entityManager->deleteUserByEmail($email);
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

  /**
   * Delete a file entity by URI.
   */
  public function deleteFileByUri(string $uri): void {
    try {
      $this->entityManager->deleteFileByUri($uri);
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

  /**
   * Create a user with a given role and fields and then log in as that user.
   *
   * @phpstan-param string[] $fields
   */
  public function loginAsUserByRole(string $roles, array $fields = []): void {
    $user = \array_map(static function ($value) {
      return $value;
    }, $fields);

    $user_created = $this->userCreate($user);
    $this->entityManager->addUserToRoles($user_created, $roles);

    $this->loginAs($user_created->name, $user_created->pass);
  }

  /**
   * @phpstan-param string[] $permissions
   * @phpstan-param string[] $fields
   */
  public function loginAsUserWithPermissions(array $permissions, array $fields = []): void {
    $role = $this->roleCreate($permissions);

    $this->loginAsUserByRole($role, $fields);
  }

  /**
   * Set the password for a user.
   */
  public function setUserPassword(UserInterface $user, string $password): void {
    try {
      $user->setPassword($password)->save();
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

  /**
   * Wait for a JavaScript condition to be true.
   */
  public function waitOnJavascript(int $timeout, string $condition): bool {
    $script = 'return (' . \rtrim($condition, " \t\n\r;") . ');';

    try {
      $start = \microtime(TRUE);
      $end = $start + $timeout / 1000.0;
      do {
        $result = self::getClient()->executeScript($script);
        if ($result === NULL) {
          break;
        }
        \usleep(10000);
      } while (\microtime(TRUE) < $end);

      return (bool) $result;
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

  /**
   * Wait for all AJAX requests to finish.
   */
  public function waitForAjaxToFinish(int $timeout = 1000): bool {
    $condition = <<<JS
    (function() {
      function isAjaxing(instance) {
        return instance && instance.ajaxing === true;
      }
      var d7_not_ajaxing = true;
      if (typeof Drupal !== 'undefined' && typeof Drupal.ajax !== 'undefined' && typeof Drupal.ajax.instances === 'undefined') {
        for(var i in Drupal.ajax) { if (isAjaxing(Drupal.ajax[i])) { d7_not_ajaxing = false; } }
      }
      var d8_not_ajaxing = (typeof Drupal === 'undefined' || typeof Drupal.ajax === 'undefined' || typeof Drupal.ajax.instances === 'undefined' || !Drupal.ajax.instances.some(isAjaxing))
      return (
        // Assert no AJAX request is running (via jQuery or Drupal) and no
        // animation is running.
        (typeof jQuery === 'undefined' || jQuery.hasOwnProperty('active') === false || (jQuery.active === 0 && jQuery(':animated').length === 0)) &&
        d7_not_ajaxing && d8_not_ajaxing
      );
    }());
JS;

    return $this->waitOnJavascript($timeout, $condition);
  }

  /**
   * Wait for a DOM element to appear.
   *
   * @param string $selector
   *   The CSS selector.
   *
   * @throws \Facebook\WebDriver\Exception\NoSuchElementException
   * @throws \Facebook\WebDriver\Exception\TimeoutException
   */
  public function waitForDomElement(string $selector): ?WebDriverElement {
    self::getClient()->wait()->until(
      static function (WebDriver $driver) use ($selector) {
        try {
          return $driver->findElement(WebDriverBy::cssSelector($selector));
        }
        catch (StaleElementReferenceException | NoSuchElementException $e) {
          return NULL;
        }
      },
    );

    return NULL;
  }

  /**
   * Expand the toolbar.
   */
  public function expandToolbar(): void {
    self::getClient()->getCrawler()->filter('.toolbar-menu__trigger')->click();
  }

  /**
   * Select a checkbox for a given data-drupal-selector.
   *
   * @param string $fieldset_label
   *   The fieldset's label which contains the checkbox.
   * @param string $checkbox_label
   *   The checkbox's label.
   */
  public function selectCheckbox(string $fieldset_label, string $checkbox_label): void {
    $selector = \sprintf(
      ".//span[@class[contains(., 'fieldset__label')] and text()[contains(., '%s')]]/ancestor::fieldset//label[text()[contains(., '%s')]]/parent::div//input[@class[contains(., 'form-checkbox')]]",
      $fieldset_label,
      $checkbox_label,
    );

    $this->waitForElementToBeClickable($selector)->click();
  }

  /**
   * Click on a field settings cog button.
   *
   * @param string $label
   *   The field's label.
   */
  public function clickFieldCogButton(string $label): void {
    $selector = \sprintf(
      ".//div[@class='tabledrag-cell-content__item' and text()[contains(., '%s')]]/ancestor::tr//input[@class[contains(., 'field-plugin-settings-edit')]]",
      $label,
    );

    $this->waitForElementToBeClickable($selector)->click();
  }

  /**
   * Wait for an element to be clickable.
   *
   * Sometimes the element you're trying to click is being covered by another
   * element, specifically a sticky region div. This method will scroll the
   * element into view and wait for it to be clickable.
   *
   * @param string $selector
   *   The xPath selector.
   *
   * @return \Facebook\WebDriver\WebDriverElement
   *   The element.
   */
  public function waitForElementToBeClickable(string $selector): WebDriverElement {
    $web_driver = WebDriverBy::xpath($selector);

    try {
      $element = self::getClient()->findElement($web_driver);
      self::getClient()->executeScript('arguments[0].scrollIntoView(true);', [$element]);
      self::getClient()->wait()->until(
        WebDriverExpectedCondition::elementToBeClickable($web_driver),
      );
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }

    return $element;
  }

  /**
   * Fetch JSON data from a given URL using JavaScript.
   *
   * @param string $url
   *   The URL to fetch the JSON data from.
   *
   * @return array<string, mixed>|null
   *   The fetched JSON data or NULL if an error occurs.
   */
  public function fetchJson(string $url): ?array {
    try {
      return self::getClient()->executeAsyncScript(
        <<<JS
                const fetchJSON = async url => {
                  const response = await fetch(url);
                  if (!response.ok) {
                    throw new Error(response.statusText);
                  }
                  return response.json();
                };
                
                const callback = arguments[arguments.length - 1];
                
                fetchJSON(arguments[0]).then(data => {
                  callback(data); 
                }).catch(error => {
                  callback({error: error.message});
                });
        JS,
        [$url],
      );
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }
  }

}
