<?php

declare(strict_types=1);

namespace Drupal\Tests\panther\FunctionalJavascript;

use Drupal\Driver\DrupalDriver;
use Drupal\panther\Drupal\Manager\DrupalUserManager;
use Drupal\panther\Drupal\Manager\EntityManager;
use Drupal\panther\Settings;
use Facebook\WebDriver\Exception\Internal\IOException;
use PHPUnit\Framework\TestCase;

class PantherTestCase extends TestCase {

  use AccessibilityTrait;
  use DomCrawlerAssertionsTrait;
  use DrupalAssertionsTrait;
  use DrupalTrait;
  use PantherTrait;

  protected DrupalDriver $driver;

  protected EntityManager $entityManager;

  protected Settings $settings;

  /**
   * @phpstan-ignore missingType.iterableValue,missingType.parameter,method.parentMethodFinalByPhpDoc
   */
  public function __construct(
    ?string $name = NULL,
    array $data = [],
    $dataName = '',
  ) {
    // @phpstan-ignore arguments.count,argument.type
    parent::__construct($name, $data, $dataName);
    $this->setRunTestInSeparateProcess(TRUE);
  }

  protected function setUp(): void {
    parent::setUp();

    $drupal_root = \getenv('PANTHER_DRUPAL_ROOT');
    $drupal_host = \getenv('PANTHER_DRUPAL_HOST');

    if ($drupal_root === FALSE || $drupal_host === FALSE) {
      throw new \RuntimeException(
        'PANTHER_DRUPAL_ROOT and PANTHER_DRUPAL_HOST environment variables must be set.',
      );
    }

    $this->driver = new DrupalDriver($drupal_root, $drupal_host);
    $this->driver->setCoreFromVersion();

    // Bootstrap Drupal.
    $this->driver->bootstrap();

    /** @var \Drupal\Core\Database\Transaction\TransactionManagerBase $transaction_manager */
    $transaction_manager = \Drupal::database()->transactionManager();

    /** @var \Drupal\user\UserStorage $user_storage */
    $user_storage = \Drupal::entityTypeManager()->getStorage('user');

    $this->entityManager = new EntityManager(
      driver: $this->driver,
      userManager: new DrupalUserManager(),
      transactionManager: $transaction_manager,
      userStorage: $user_storage,
    );

    $this->settings = new Settings();

    $this->createDirectoryIfNotExists($this->settings->accessibilityReportsDir);
    $this->createDirectoryIfNotExists($this->settings->screenshotsDir);
  }

  protected function tearDown(): void {
    parent::tearDown();

    $this->takeScreenshotIfTestFailed();

    // Build a new client instance for each test.
    self::getClient()->quit();
    self::resetClient();

    // Clean up.
    $this->entityManager->cleanAll();
  }

  private function createDirectoryIfNotExists(string $directory_path): void {
    if (!\file_exists($directory_path)) {
      if (!\mkdir($directory_path, 0777, TRUE) && !\is_dir($directory_path)) {
        throw IOException::forFileError(
          'Directory cannot be not created',
          $directory_path,
        );
      }
    }
  }

}
