<?php

declare(strict_types=1);

namespace Drupal\Tests\panther\FunctionalJavascript;

use Facebook\WebDriver\Remote\DesiredCapabilities;
use Symfony\Component\Panther\Client;

trait PantherTrait {

  protected static function getClient(bool $reset = FALSE): Client {
    static $client;

    $selenium_host = \getenv('PANTHER_SELENIUM_HOST');
    $drupal_host = \getenv('PANTHER_DRUPAL_HOST');

    if ($selenium_host === FALSE || $drupal_host === FALSE) {
      throw new \RuntimeException(
        'PANTHER_SELENIUM_HOST and PANTHER_DRUPAL_HOST environment variables must be set.',
      );
    }

    if (!$client || $reset) {
      // '--disable-dev-shm-usage' is required to avoid the "session deleted
      // because of page crash" error.
      $capabilities = new DesiredCapabilities(
        [
          'browserName' => 'chrome',
          'goog:chromeOptions' => [
            'args' => [
              '--no-sandbox',
              '--disable-dev-shm-usage',
              '--disable-gpu',
            ],
          ],
        ],
      );

      $connection_timeout = \getenv('PANTHER_CONNECTION_TIMEOUT_MS');
      $request_timeout = \getenv('PANTHER_REQUEST_TIMEOUT_MS');

      if ($connection_timeout === FALSE) {
        $connection_timeout = 5000;
      }

      if ($request_timeout === FALSE) {
        $request_timeout = 5000;
      }

      self::waitUntilDrupalIsAvailable($drupal_host, (int) $connection_timeout);

      $client = Client::createSeleniumClient(
        $selenium_host,
        $capabilities,
        $drupal_host,
        [
          'connection_timeout_in_ms' => (int) $connection_timeout,
          'request_timeout_in_ms' => (int) $request_timeout,
        ],
      );
    }

    return $client;
  }

  private static function resetClient(): void {
    self::getClient(TRUE);
  }

  public function takeScreenshotIfTestFailed(): void {
    // @phpcs:disable Drupal.Classes.FullyQualifiedNamespace.UseStatementMissing
    if (\class_exists(\PHPUnit\Runner\BaseTestRunner::class) && \method_exists(
        $this,
        'getStatus',
      )) {
      // PHPUnit <10 TestCase.
      $status = $this->getStatus();
      $isError = \PHPUnit\Runner\BaseTestRunner::STATUS_FAILURE === $status;
      $isFailure = \PHPUnit\Runner\BaseTestRunner::STATUS_ERROR === $status;
      // @phpcs:enable Drupal.Classes.FullyQualifiedNamespace.UseStatementMissing
    }
    // @phpstan-ignore-next-line
    elseif (\method_exists($this, 'status')) {
      // PHPUnit 10 TestCase.
      $status = $this->status();
      $isError = $status->isError();
      $isFailure = $status->isFailure();
    }
    else {
      // Symfony WebTestCase.
      return;
    }
    if ($isError || $isFailure) {
      $type = $isError ? 'error' : 'failure';
      $this->takeScreenshot($type, $this->toString());
    }
  }

  public function takeScreenshot(string $type, string $test): void {
    $screenshot_dir = $this->settings->screenshotsDir;

    // Ensure the directory exists.
    if (!\is_dir($screenshot_dir)) {
      return;
    }

    // Ensure the directory is writable.
    if (!\is_writable($screenshot_dir)) {
      return;
    }

    $screenshot_path = \sprintf(
      '%s/%s_%s.png',
      $screenshot_dir,
      $type,
      \strtr($test, ['\\' => '-', ':' => '_']),
    );
    self::getClient()->takeScreenshot($screenshot_path);
  }

  protected function goToPage(string $path): void {
    self::getClient()->request('GET', $path);
  }

  protected function maximizeWindow(): void {
    self::getClient()->manage()->window()->maximize();
  }

  /**
   * @phpstan-param string[] $fields
   */
  protected function submitForm(string $button, array $fields): void {
    self::getClient()->submitForm($button, $fields);
  }

  protected function clickLink(string $string): void {
    self::getClient()->clickLink($string);
  }

  protected function clickButton(string $string): void {
    self::getClient()->getCrawler()->selectButton($string)->click();
  }

  /**
   * Waits until Drupal is available at the specified URL.
   *
   * @param string $url
   *   The URL to check.
   * @param int $connection_timeout
   *   The connection timeout in seconds.
   * @param int $max_attempts
   *   The maximum number of attempts to check the URL.
   * @param int $wait_seconds
   *   The number of seconds to wait between attempts.
   *
   * @throws \RuntimeException
   *   If Drupal is not available after the specified attempts.
   */
  protected static function waitUntilDrupalIsAvailable(string $url, int $connection_timeout = 2, int $max_attempts = 10, int $wait_seconds = 2): void {
    $client = \Drupal::service('http_client_factory')->fromOptions([
      'timeout' => $connection_timeout,
      'allow_redirects' => FALSE,
      'headers' => [
        'User-Agent' => 'PantherHealthCheck/1.0',
      ],
    ]);

    $url = \rtrim($url, '/') . '/';

    for ($i = 0; $i < $max_attempts; $i++) {
      try {
        $response = $client->head($url);
        $status_code = $response->getStatusCode();

        if ($status_code >= 200 && $status_code < 500) {
          // If we get a 2xx or 3xx response, Drupal is available.
          return;
        }
      }
      catch (\Exception $e) {
        // If we catch an exception, it means the request failed.
        // We can log the error or handle it as needed.
        // For now, we just continue to the next attempt.
      }

      \sleep($wait_seconds);
    }

    throw new \RuntimeException("Unable to reach Drupal at {$url} after {$max_attempts} attempts.");
  }

}
