<?php

declare(strict_types=1);

namespace Drupal\panther\Phpunit\Context;

use Drupal\panther\Phpunit\Enums\TestStatus;
use Drupal\panther\Phpunit\Output\FormatterInterface;
use Drupal\panther\Phpunit\Output\PrinterInterface;
use Drupal\panther\Phpunit\Output\ResourceUsageFormatter;
use PHPUnit\Event\Telemetry\Duration;
use PHPUnit\Event\Telemetry\Info;

class TestContext {

  protected int $totalTests;

  protected int $currentTestNumber = 0;

  /**
   * @var array<string, TestClassInformation>
   */
  protected array $testClasses = [];

  protected ?string $currentTestFile = NULL;

  protected ?string $currentTestMethod = NULL;

  public function __construct(
    protected readonly string $basePath,
    protected readonly FormatterInterface $formatter,
    protected readonly PrinterInterface $printer,
  ) {}

  public function setTotalTests(int $total): void {
    $this->totalTests = $total;
  }

  public function markRunnerStarted(): void {
    $this->printer->line('');
  }

  public function markTestStarted(string $file, string $className, string $method): void {
    $this->incrementTestNumber();

    $cleanFileName = $this->afterString($file, $this->basePath);

    $this->currentTestMethod = $method;

    $this->handleFirstTestForClass($cleanFileName, $className);

    $this->testClasses[$cleanFileName]->tests[$method] = new TestInformation(
      $method,
      $this->currentTestNumber,
      TestStatus::Running,
    );
  }

  public function markTestFinished(Duration $duration): void {
    $this->currentTestInformation()->duration = $duration;

    $this->printer->line(
      $this->formatter->testFinish(
        $this->currentTestInformation(),
      ),
    );
  }

  public function markRunnerFinished(Info $telemetry): void {
    $resource_usage = (new ResourceUsageFormatter())->resourceUsage($telemetry->durationSinceStart(), $telemetry->memoryUsageSinceStart());
    $this->printer->line("\n" . $resource_usage);

    $this->printer->finalize();
  }

  public function markTestPassed(string $message): void {
    $this->currentTestInformation()->status = TestStatus::Passed;
    $this->currentTestInformation()->message = $message;
  }

  public function markTestErrored(string $message): void {
    $this->currentTestInformation()->status = TestStatus::Errored;
    $this->currentTestInformation()->message = $message;
  }

  public function markTestWarning(string $message): void {
    $this->currentTestInformation()->status = TestStatus::Warning;
    $this->currentTestInformation()->message = $message;
  }

  public function markTestFailed(string $message): void {
    $this->currentTestInformation()->status = TestStatus::Failed;
    $this->currentTestInformation()->message = $message;
  }

  public function markTestIncomplete(string $message): void {
    $this->currentTestInformation()->status = TestStatus::Incomplete;
    $this->currentTestInformation()->message = $message;
  }

  /**
   * Mark a test as risky.
   *
   * A test passes or fails first, then it may be considered risky.
   * Only set the status to risky if we passed the test.
   */
  public function markTestRisky(string $message): void {
    if ($this->currentTestInformation()->status !== TestStatus::Passed) {
      return;
    }

    $this->currentTestInformation()->status = TestStatus::Risky;
    $this->currentTestInformation()->message = $message;
  }

  public function markPhpDeprecation(string $file, int $line, string $message): void {
    $this->printer->line(
      $this->formatter->deprecation($file, $line, $message),
    );
  }

  public function markUnexpectedOutput(string $output): void {
    $lines = \explode("\n", \trim($output));

    foreach ($lines as $line) {
      $this->printer->line(
        $this->formatter->unexpectedOutput($line),
      );
    }
  }

  public function markBeforeMethodErrored(string $message): void {
    $this->printer->line(
      $this->formatter->errorBeforeMethod($message),
    );
  }

  protected function handleFirstTestForClass(string $cleanFileName, string $className): void {
    if (\array_key_exists($cleanFileName, $this->testClasses)) {
      return;
    }

    $this->currentTestFile = $cleanFileName;

    $this->testClasses[$cleanFileName] = new TestClassInformation($cleanFileName, $className);

    $this->printer->line(
      $this->formatter->testClassStart(
        $this->testClasses[$cleanFileName],
        $this->currentTestNumber,
        $this->totalTests,
      ),
    );
  }

  protected function currentTestInformation(): TestInformation {
    return $this->testClasses[$this->currentTestFile]->tests[$this->currentTestMethod];
  }

  protected function incrementTestNumber(): void {
    $this->currentTestNumber++;
  }

  protected function afterString(string $haystack, string $needle): string {
    if (!\str_starts_with($haystack, $needle)) {
      return $haystack;
    }

    return \substr($haystack, \strlen($needle));
  }

}
