<?php

declare(strict_types=1);

namespace Drupal\panther\Phpunit\Output;

use Drupal\panther\Phpunit\Context\TestClassInformation;
use Drupal\panther\Phpunit\Context\TestInformation;
use Drupal\panther\Phpunit\Enums\TestStatus;
use PHPUnit\Event\Telemetry\Duration;
use PHPUnit\Util\Color;

class Formatter implements FormatterInterface {

  public function testClassStart(
    TestClassInformation $class,
    int $current,
    int $total,
  ): string {
    $padNumberTo = \strlen((string) $total);

    return Color::colorize('fg-white', '[ ')
      . Color::colorize('fg-yellow', \str_pad((string) $current, $padNumberTo, ' ', STR_PAD_LEFT))
      . Color::colorize(
        'fg-white',
        ' / ' . $total . ' ]   ' . $class->class,
      );
  }

  public function testFinish(TestInformation $test): string {
    $specialState = $this->getSpecialStateString($test->status);

    return $this->indent()
      . Color::colorize(
        $this->getColorStringForState($test->status),
        $this->getSymbolForState($test->status)
        . ' '
        . NameUtil::friendly($test->method)
        . $this->getMessage($test->message)
        . (
        $specialState !== ''
          ? ' <' . $specialState . '>'
          : ''
        ),
      )
      . $this->makeSlowTestSuffix($test->duration);
  }

  protected function getMessage(string $message): string {
    if ($message === '') {
      return '';
    }

    return \sprintf(' (%s)', $message);
  }

  protected function makeSlowTestSuffix(?Duration $duration): ?string {
    if ($duration === NULL) {
      return NULL;
    }

    $seconds = $duration->nanoseconds() / 100000000;

    if ($seconds < $this->getSlowTestThresholdInSeconds()) {
      return NULL;
    }

    $color = $seconds > $this->getVerySlowTestThresholdInSeconds()
      ? 'fg-red'
      : 'fg-yellow';

    return Color::colorize($color, \sprintf(' [%.3fs]', $seconds));
  }

  public function unexpectedOutput(string $output): string {
    return '> ' . $output;
  }

  public function errorBeforeMethod(string $message): string {
    $output = [];

    $lines = \explode("\n", $message);
    $output[] = $this->indent() . Color::colorize('fg-red', 'Error before method: ');
    foreach ($lines as $line) {
      $output[] = $this->indent(2) . Color::colorize('fg-red', $line);
    }

    return \implode("\n", $output);
  }

  public function deprecation(string $file, int $line, string $message): string {
    return $this->indent() . $file . ':' . $line . ': ' . $message;
  }

  protected function getSpecialStateString(TestStatus $status): string {
    $noSpecialStringFor = [
      TestStatus::Passed,
      TestStatus::Failed,
      TestStatus::Errored,
      TestStatus::Running,
    ];

    if (\in_array($status, $noSpecialStringFor, TRUE)) {
      return '';
    }

    return $status->friendly();
  }

  protected function getSymbolForState(TestStatus $status): string {
    return match ($status) {
      TestStatus::Passed => '✓',
      TestStatus::Incomplete,
      TestStatus::Skipped,
      TestStatus::Risky,
      TestStatus::Warning => '-',
      TestStatus::Errored,
      TestStatus::Failed => 'x',
      TestStatus::Running => '.',
    };
  }

  protected function getColorStringForState(TestStatus $status): string {
    return match ($status) {
      TestStatus::Passed => 'fg-green',
      TestStatus::Incomplete,
      TestStatus::Skipped,
      TestStatus::Warning => 'fg-yellow',
      TestStatus::Errored,
      TestStatus::Failed => 'fg-red',
      TestStatus::Risky => 'fg-magenta',
      TestStatus::Running => 'fg-white',
    };
  }

  protected function indent(int $count = 1): string {
    return \str_repeat('    ', $count);
  }

  protected function getSlowTestThresholdInSeconds(): float {
    return 0.5;
  }

  protected function getVerySlowTestThresholdInSeconds(): float {
    return 1.0;
  }

}
