<?php

declare(strict_types=1);

namespace Drupal\Tests\panther\FunctionalJavascript;

use Drupal\panther\Axe\AccessibilityReportDumper;

trait AccessibilityTrait {

  /**
   * Run accessibility test using Axe on a specific path.
   */
  public function runAccessibilityTestOnPath(
    string $path,
    int $timeout = 1000,
  ): void {
    $this->goToPage($path);
    $this->runAccessibilityTest($timeout);
  }

  /**
   * Run accessibility test using Axe.
   */
  public function runAccessibilityTest(int $timeout = 1000): void {
    $function = $this->getJavascriptFunction();
    $accessibility_reports_dir = $this->settings->accessibilityReportsDir;
    $dump_accessibility_reports = $this->settings->dumpAccessibilityReports;

    try {
      $start = \microtime(TRUE);
      $end = $start + $timeout / 1000.0;
      do {
        $result = self::getClient()->executeScript($function);
        if ($result !== NULL) {
          break;
        }
        \usleep(10000);
      } while (\microtime(TRUE) < $end);
    }
    catch (\Exception $e) {
      self::fail($e->getMessage());
    }

    if ($dump_accessibility_reports) {
      $accessibility_report_dumper = new AccessibilityReportDumper(
        [
          'accessibilityIndex' => $this->calculateAccessibilityIndex($result),
        ],
      );
      $report_file_path = \sprintf(
        '%s/%s.html',
        $accessibility_reports_dir,
        $this->getSanitizedTestName(),
      );
      $accessibility_report_dumper->dump($result, $report_file_path);
    }

    $this->assertOnResult($result);
  }

  /**
   * Get the Axe tags string for the accessibility test.
   */
  private function getAxeTagsString(): string {
    $axe_tags = $this->settings->axeTags;
    $axe_best_practices = $this->settings->axeBestPractices;

    $tags = [];
    if ($axe_best_practices) {
      $tags[] = 'best-practice';
    }
    if (\count($axe_tags) > 0) {
      $tags = \array_merge($tags, $axe_tags);
    }
    $tags = \array_unique($tags);
    $tags = \array_values($tags);

    return \implode("', '", $tags);
  }

  /**
   * Get the JavaScript function used to inject Axe and run accessibility tests.
   */
  private function getJavascriptFunction(): string {
    $axe_script_url = $this->settings->axeScriptUrl;
    $axe_tags_string = $this->getAxeTagsString();

    return <<<JS
async function runAxe() {
  return new Promise((resolve, reject) => {
    var script = document.createElement('script');
    script.src = '$axe_script_url';
    script.onload = function () {
      axe
        .run({
          runOnly: {
            type: 'tag',
            values: ['$axe_tags_string']
          }
        })
        .then(results => {
          resolve(results);
        })
        .catch(err => {
          reject(err);
        });
    };
    script.onerror = function () {
      reject(new Error('Failed to load $axe_script_url'));
    };
    document.head.appendChild(script);
  });
}

return await runAxe();
JS;
  }

  /**
   * Assert the result of the accessibility test.
   *
   * @phpstan-param array<array-key, mixed> $result
   */
  private function assertOnResult(array $result): void {
    $fail_on_axe_error = $this->settings->failOnAxeError;

    $violations = $result['violations'];

    if (\count($violations) > 0) {
      $messages = [];
      foreach ($violations as $violation) {
        $messages[] = \sprintf(
          'Violation: %s, Impact: %s, Nodes: %s',
          $violation['description'],
          $violation['impact'],
          \implode(
            ', ',
            \array_map(static fn($node) => $node['html'], $violation['nodes']),
          ),
        );
      }
      if ($fail_on_axe_error) {
        self::fail(
          'Accessibility violations found: ' . \implode('; ', $messages),
        );
      }
      else {
        self::markTestIncomplete(
          'Accessibility violations found: ' . \implode('; ', $messages),
        );
      }
    }
    else {
      self::assertCount(0, $violations);
    }
  }

  /**
   * @phpstan-param array<array-key, mixed> $result
   */
  private function calculateAccessibilityIndex(array $result): float {
    $impact_weights = [
      'critical' => 10,
      'serious' => 5,
      'moderate' => 3,
      'minor' => 1,
    ];
    $pass_weight = 0.1;
    $base_score = 100;
    $penalty_score = 0;
    $success_score = 0;

    if (isset($result['violations']) && \is_array($result['violations'])) {
      foreach ($result['violations'] as $violation) {
        $impact = $violation['impact'] ?? 'minor';
        $num_nodes = \count($violation['nodes']);
        $penalty_score += $num_nodes * ($impact_weights[$impact] ?? 0);
      }
    }

    if (isset($result['incomplete']) && \is_array($result['incomplete'])) {
      foreach ($result['incomplete'] as $incomplete) {
        $impact = $incomplete['impact'] ?? 'minor';
        $num_nodes = \count($incomplete['nodes']);
        $penalty_score += $num_nodes * ($impact_weights[$impact] ?? 0);
      }
    }

    if (isset($result['passes']) && \is_array($result['passes'])) {
      foreach ($result['passes'] as $pass) {
        $num_nodes = \count($pass['nodes']);
        $success_score += $num_nodes * $pass_weight;
      }
    }

    $min_original = 0;
    $max_original = $base_score + $success_score;
    $min_target = 0;
    $max_target = 100;

    $accessibility_index = $base_score - $penalty_score + $success_score;
    $rounded = (int) \round($accessibility_index);

    return (($rounded - $min_original) / ($max_original - $min_original)) * ($max_target - $min_target) + $min_target;
  }

  /**
   * Convert the test name to a string suitable for a file name.
   */
  private function getSanitizedTestName(): string {
    $sanitized_name = \preg_replace(
      pattern: '/[^a-zA-Z0-9_\-\s]/',
      replacement: '',
      subject: $this->toString(),
    );

    // In case the test name is empty or contains only invalid characters,
    // we default to a generic name.
    if ($sanitized_name === NULL) {
      $sanitized_name = 'accessibility-test';
    }

    return \str_replace(' ', '-', $sanitized_name);
  }

}
