<?php

declare(strict_types=1);

use Drupal\Tests\panther\FunctionalJavascript\PantherTestCase;
use PHPUnit\Framework\Attributes\Group;

class HomePageTest extends PantherTestCase {

  public function testHomePage(): void {
    $this->maximizeWindow();
    $this->goToPage('/');
    self::assertPageContains('Super easy vegetarian pasta bake');
  }

  public function testLoginPage(): void {
    $this->maximizeWindow();
    $this->goToPage('/user/login');
    self::assertPageContains('Log in');
    $this->takeScreenshot('testLoginPage', 'login');
  }

  public function testLogin(): void {
    $this->loginAsUserByRole('author', ['mail' => 'author@example.com']);
    self::assertPageContains('Member for');
  }

  public function testArticleCreate(): void {
    $node = [
      'title' => 'Test node',
      'type' => 'article',
      'field_body:value' => 'This is a test node.',
      'field_body:format' => 'full_html',
      'uid' => 1,
      'status' => 1,
      'moderation_state' => 'published',
      'path' => '/test-node',
    ];

    $this->nodeCreate($node);

    $this->goToPage('/test-node');
    self::assertPageContains('Test node');
    self::assertPageContains('This is a test node.');
  }

  public function testRecipeCreate(): void {
    $node = [
      'title' => 'Recipe1',
      'type' => 'recipe',
      'field_preparation_time' => '10',
      'field_cooking_time' => '20',
      'field_difficulty' => 'easy',
      'field_ingredients' => 'Ingredient1, Ingredient2',
      'uid' => 1,
      'status' => 1,
      'moderation_state' => 'published',
      'path' => '/recipe1',
    ];

    $this->nodeCreate($node);

    $this->goToPage('/recipe1');
    self::assertPageContains('Recipe1');
    self::assertPageContains('10 minutes');
    self::assertPageContains('20 minutes');
    self::assertPageContains('Easy');
    self::assertPageContains('Ingredient1');
    self::assertPageContains('Ingredient2');
  }

  public function testUserCreate(): void {
    $user = $this->userCreate([
      'name' => 'testuser',
    ]);

    $this->loginAs($user->name, $user->pass);
    $this->goToPage('/user/' . $user->uid);
    $this->takeScreenshot('testUserCreate', 'login');
    self::assertPageContains('testuser');
  }

  public function testRoleCreate(): void {
    $permissions = [
      'access administration pages',
      'access content',
      'create article content',
    ];

    $role_name = $this->roleCreate($permissions);

    $this->loginAsUserByRole($role_name);

    $this->goToPage('/node/add/article');
    self::assertPageContains('Create Article');
  }

  public function testTermCreate(): void {
    $term = $this->termCreate([
      'name' => 'Test term',
      'type' => 'tags',
      'vocabulary_machine_name' => 'tags',
      'status' => 1,
    ]);

    $this->goToPage('/taxonomy/term/' . $term->tid);
    self::assertPageContains('Test term');
  }

  public function testLanguageCreate(): void {
    $language = [
      'langcode' => 'fr',
    ];

    $this->languageCreate($language);

    $this->loginAsUserWithPermissions([
      'access administration pages',
      'administer languages',
      'administer site configuration',
    ]);

    $this->goToPage('/admin/config/regional/language');
    self::assertPageContains('French');
  }

  public function testMenuCreate(): void {
    $menu = [
      'label' => 'Test Menu',
      'description' => 'A test menu for functional tests.',
    ];

    $this->menuCreate($menu);

    $this->loginAsUserWithPermissions([
      'administer menu',
    ]);

    $this->goToPage('/admin/structure/menu/manage/test_menu');
    self::assertPageContains('Test Menu');
  }

  public function testMenuLinkContentCreate(): void {
    $menu_link_content = [
      'title' => 'Test menu item 1',
      'uri' => 'internal:/test_menu_item_1',
    ];

    $this->menuLinkContentCreate('Main navigation', $menu_link_content);

    $this->loginAsUserWithPermissions([
      'administer menu',
    ]);

    $this->goToPage('/admin/structure/menu/manage/main');
    self::assertPageContains('Test menu item 1');
  }

  #[Group('accessibility')]
  public function testAccessibility(): void {
    $this->runAccessibilityTestOnPath('/');
  }

}
