<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Manages Administrative Title fields and field weights for paragraph forms.
 *
 * Contains in particular code for adjusting field weights.
 *
 * @package Drupal\paragraph_group\Paragroup
 */
class ParagroupFieldManager {

  public function __construct(
    protected EntityDisplayRepositoryInterface $entityDisplayRepository,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {

    return new static(
      $container->get('entity_display.repository')
    );

  }

  /**
   * Gets the field name of the lowest weight field from bundle fields.
   *
   * @param array $fields
   *   Array of field configuration data.
   *
   * @return string|false
   *   The field name with the lowest weight, or FALSE if no fields found.
   */
  private function getLowestWeight($fields) {

    $lowest_weight = FALSE;
    $lw_field_name = FALSE;

    foreach ($fields as $field_name => $field) {

      if ($lowest_weight === FALSE ||
         $field['weight'] < $lowest_weight
      ) {
        $lowest_weight = $field['weight'];
        $lw_field_name = $field_name;
      }

    }

    return $lw_field_name;

  }

  /**
   * Filters empty keys from fields.
   *
   * @param array $fields
   *   Array of field configuration data.
   *
   * @return array
   *   The filtered fields array with empty keys removed.
   */
  private static function filterFields($fields) {

    if (isset($fields[""])) {
      unset($fields[""]);
    }

    return $fields;

  }

  /**
   * Takes a list of fields and returns them reordered by weight.
   *
   * @param array $fields
   *   Array of field configuration data.
   *
   * @return array
   *   Array of fields ordered by weight with field_name added to each field.
   */
  private function orderFieldsByWeight($fields) {

    $fields = $this->filterFields($fields);
    $fields_by_weight = [];

    while (!empty($fields)) {

      $lw_field_name = $this->getLowestWeight($fields);
      $field = $fields[$lw_field_name];

      $field['field_name'] = $lw_field_name;
      $fields_by_weight[] = $field;
      unset($fields[$lw_field_name]);

    }

    return $fields_by_weight;

  }

  /**
   * Adjusts field weights to a discrete integer setting.
   *
   * E.g. -1, 4, 100, ... becomes 1, 2, 3, ... if the $start_weight is 1.
   *
   * @param array $fields_by_weight
   *   Array of fields ordered by weight.
   * @param int $start_weight
   *   The starting weight value for the first field.
   *
   * @return array
   *   Array of fields with rationalized weights.
   */
  private static function orderFieldsByRationalWeight(
    $fields_by_weight,
    $start_weight,
  ) {

    $i = $start_weight;
    $fields_by_rational_weight = [];

    foreach ($fields_by_weight as $field) {

      $field['weight'] = $i;
      $fields_by_rational_weight[$i] = $field;
      $i++;

    }

    return $fields_by_rational_weight;

  }

  /**
   * Returns field array elements to their original state with updated weights.
   *
   * Ensures the fields are readable by the Drupal field update function.
   *
   * @param array $fields_by_rational_weight
   *   Array of fields with rationalized weights.
   *
   * @return array
   *   Array of fields in original format with updated weights.
   */
  private static function resetFieldsWithNewWeights(
    $fields_by_rational_weight,
  ) {

    $fields = [];

    foreach ($fields_by_rational_weight as $field) {

      $field_name = $field['field_name'];
      unset($field['field_name']);
      $fields[$field_name] = $field;

    }

    return $fields;

  }

  /**
   * Updates each field containing new weight.
   *
   * @param string $entity_type
   *   The entity type machine name.
   * @param string $bundle
   *   The bundle machine name.
   * @param array $fields
   *   Array of field configuration data with updated weights.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function updateFieldWeights($entity_type, $bundle, $fields) {

    foreach ($fields as $field_name => $field) {

      $this->entityDisplayRepository
        ->getFormDisplay($entity_type, $bundle)
        ->setComponent($field_name, $field)
        ->save();

    }

    return TRUE;

  }

  /**
   * Carries out the actions to create a new field.
   *
   * Uses Drupal's FieldConfig and FieldStorageConfig classes.
   *
   * @param array $field_storage_config
   *   Configuration array for field storage.
   * @param array $field_config
   *   Configuration array for field instance.
   *
   * @return bool
   *   TRUE if field config was created, FALSE if it already existed.
   */
  private static function createFieldActionsConfig(
    $field_storage_config,
    $field_config,
  ) {

    $bundle = $field_config['bundle'];
    $field_name = $field_config['field_name'];
    $entity_type = $field_config['entity_type'];

    $load_field_storage_config =
      FieldStorageConfig::loadByName($entity_type, $field_name);

    if (!$load_field_storage_config) {
      FieldStorageConfig::create($field_storage_config)->save();
    }

    $load_field_config =
      FieldConfig::loadByName($entity_type, $bundle, $field_name);

    $field_config_created = FALSE;

    if (!$load_field_config) {

      FieldConfig::create($field_config)->save();
      $field_config_created = TRUE;

    }

    return $field_config_created;

  }

  /**
   * Creates field and updates form display to enable it.
   *
   * Main field creation is handled by createFieldActionsConfig; this function
   * additionally updates the form display so the field is not initially in the
   * disabled fields section.
   *
   * @param array $field_storage_config
   *   Configuration array for field storage.
   * @param array $field_config
   *   Configuration array for field instance.
   * @param array $form_options
   *   Form display options for the field.
   *
   * @return bool
   *   TRUE if field was created and form display updated, FALSE otherwise.
   */
  private function createFieldActions(
    $field_storage_config,
    $field_config,
    $form_options,
  ) {

    $field_config_created = $this->createFieldActionsConfig(
      $field_storage_config, $field_config
    );

    if ($field_config_created) {

      $bundle = $field_config['bundle'];
      $field_name = $field_config['field_name'];
      $entity_type = $field_config['entity_type'];

      // Manage form display - enables the field.
      $this->entityDisplayRepository
        ->getFormDisplay($entity_type, $bundle)
        ->setComponent($field_name, $form_options)
        ->save();

      return TRUE;

    }

    return FALSE;

  }

  /**
   * Prepares field weights of a bundle by setting them to discrete numbers.
   *
   * Previously the weight integers are 'all over the shop', with any difference
   * permitted between the field weights. Rationalising and resetting the field
   * weights is necessary so that the Administrative Title field is the first
   * field in a Paragraph, and also so that the fields are correctly allocated
   * to Field Groups by ParagroupFieldGroupManager.
   *
   * @param string $entity_type
   *   The entity type machine name.
   * @param string $bundle
   *   The bundle machine name.
   * @param int $start_weight
   *   The starting weight value for the first field.
   *
   * @return bool
   *   TRUE if field weights were successfully updated.
   */
  public function prepareFieldWeights(
    $entity_type,
    $bundle,
    $start_weight = 1,
  ) {

    $fields = $this->entityDisplayRepository
      ->getFormDisplay($entity_type, $bundle)
      ->getComponents();

    $fields_by_weight =
      $this->orderFieldsByWeight($fields);

    $fields_by_rational_weight =
      $this->orderFieldsByRationalWeight(
        $fields_by_weight, $start_weight
      );

    $reset_fields =
      $this->resetFieldsWithNewWeights(
        $fields_by_rational_weight
      );

    $updated_weights =
      $this->updateFieldWeights(
        $entity_type, $bundle, $reset_fields
      );

    return $updated_weights;

  }

  /**
   * Creates a field by first updating field weights.
   *
   * Updates a bundle's field weights, then creates the field.
   *
   * @param array $field_storage_config
   *   Configuration array for field storage.
   * @param array $field_config
   *   Configuration array for field instance.
   * @param array $form_options
   *   Form display options for the field.
   *
   * @return bool
   *   TRUE if field was created successfully.
   */
  public function createField(
    $field_storage_config,
    $field_config,
    $form_options,
  ) {

    $bundle = $field_config['bundle'];
    $entity_type = $field_config['entity_type'];

    $this->prepareFieldWeights($entity_type, $bundle);

    $field_created = $this->createFieldActions(
      $field_storage_config, $field_config, $form_options
    );

    return $field_created;

  }

  /**
   * Deletes a field by removing its FieldConfig info from the bundle.
   *
   * This deletes all content stored for that bundle. The field's general
   * FieldStorageConfig remains in the database.
   *
   * @param string $entity_type
   *   The entity type machine name.
   * @param string $bundle
   *   The bundle machine name.
   * @param string $field_name
   *   The field machine name to delete.
   *
   * @return bool
   *   TRUE if field was deleted, FALSE if field config was not found.
   */
  public function deleteField($entity_type, $bundle, $field_name) {

    $load_field_config =
      FieldConfig::loadByName($entity_type, $bundle, $field_name);

    if ($load_field_config) {
      $load_field_config->delete();
      return TRUE;
    }

    return FALSE;

  }

  /**
   * Creates an Administrative Title field.
   *
   * Uses the general field creation system in this class.
   *
   * @param string $bundle
   *   The paragraph bundle machine name.
   *
   * @return bool
   *   TRUE if field was created successfully.
   */
  public function createParagraphAdminTitleField($bundle) {

    $field_storage_config = [
      'field_name' => 'paragroup_admin_title',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ];

    $field_config = [
      'field_name' => 'paragroup_admin_title',
      'entity_type' => 'paragraph',
      'bundle' => $bundle,
      'label' => 'Administrative Title',
      'field_type' => 'string',
    ];

    $form_options = [
      'type' => 'string_textfield',
      'weight' => 0,
    ];

    $field_created = $this->createField(
      $field_storage_config, $field_config, $form_options
    );

    return $field_created;

  }

  /**
   * Deletes an Administrative Title field.
   *
   * Uses the general field deletion system in this class.
   *
   * @param string $bundle
   *   The paragraph bundle machine name.
   *
   * @return bool
   *   TRUE if field was deleted successfully.
   */
  public function deleteParagraphAdminTitleField($bundle) {

    $field_deleted = $this->deleteField(
      'paragraph', $bundle, 'paragroup_admin_title'
    );

    return $field_deleted;

  }

}
