<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\field\Entity\FieldConfig;
use Drupal\Core\Field\WidgetPluginManager;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides form data and checkboxes for the settings configuration form.
 *
 * @package Drupal\paragraph_group\Paragroup
 */
class ParagroupFormData {

  use StringTranslationTrait;

  public function __construct(
    TranslationInterface $string_translation,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RequestStack $requestStack,
    protected EntityDisplayRepositoryInterface $entityDisplayRepository,
    protected WidgetPluginManager $widgetManager,
    protected ParagroupFieldGroupManager $fieldGroupManager,
  ) {
    $this->setStringTranslation($string_translation);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {

    return new static(
      $container->get('string_translation'),
      $container->get('entity_type.manager'),
      $container->get('request_stack'),
      $container->get('entity_display.repository'),
      $container->get('plugin.manager.field.widget'),
      $container->get('paragraph_group.field_group_manager')
    );

  }

  /**
   * Gets complete list of Paragraph types.
   *
   * @return array
   *   Array of paragraph type info, excluding 'from_library' type.
   */
  private function getParagraphTypes() {

    $paragraph_types =
      $this->entityTypeManager->getStorage('paragraphs_type')->loadMultiple();

    if (isset($paragraph_types['from_library'])) {
      unset($paragraph_types['from_library']);
    }

    return $paragraph_types;

  }

  /**
   * Determines whether a Paragraph bundle has an Administrative Title field.
   *
   * @param string $machine_name
   *   The paragraph bundle machine name.
   *
   * @return string
   *   'Yes' if field exists, 'No' otherwise.
   */
  private function hasAdminTitleField($machine_name) {

    $field = FieldConfig::loadByName(
      'paragraph', $machine_name, 'paragroup_admin_title'
    );

    if ($field) {
      return $this->t('Yes');
    }

    return $this->t('No');

  }

  /**
   * Formats an option array for getAdminTitlesOptions.
   *
   * @param string $machine_name
   *   The paragraph bundle machine name.
   * @param array $info
   *   The paragraph type info array.
   *
   * @return array
   *   Formatted option array for the paragraph type.
   */
  private function getParagraphOption($machine_name, $info) {

    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    $path = "/admin/structure/paragraphs_type/$machine_name/fields";
    $url = $base_url . $path;
    $manage_fields = $this->t('<a href=":url" target="_blank">Manage Fields</a>', [':url' => $url]);
    $has_field = $this->hasAdminTitleField($machine_name);

    $option = [
      'paragraph_type' => ['data' => $info->label(), 'class' => 'name-field'],
      'machine_name' => ['data' => $machine_name],
      'has_field' => ['data' => $has_field],
      'manage' => ['data' => $manage_fields],
    ];

    return $option;

  }

  /**
   * Gets the Manage Form Display link for getFieldGroupsOptions.
   *
   * @param string $type
   *   The content type machine name.
   *
   * @return string
   *   HTML link to the form display management page.
   */
  private function getContentTypeOptionLabel($type) {

    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    $path = "/admin/structure/types/manage/$type/form-display";
    $url = $base_url . $path;

    $label = $this->t('<a href=":url" target="_blank">Manage Form Display</a>', [':url' => $url]);

    return $label;

  }

  /**
   * Returns list of content types.
   *
   * @return array
   *   Array of node type entities.
   */
  private function getContentTypes() {

    $types = $this->entityTypeManager
      ->getStorage('node_type')
      ->loadMultiple();

    return $types;

  }

  /**
   * Gets form display url paths for each entity type.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   *
   * @return string|null
   *   The URL to the form display page, or NULL if not supported.
   */
  private function getDetailsWidgetOptionUrl($field_id) {

    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();

    $field_parts = explode('.', $field_id);
    $entity_type = $field_parts[0];
    $bundle = $field_parts[1];

    $path = '';

    switch ($entity_type) {

      case 'node':
        $path =
          "admin/structure/types/manage/$bundle/form-display";
        break;

      case 'paragraph':
        $path =
          "admin/structure/paragraphs_type/$bundle/form-display";
        break;

      case 'taxonomy_term':
        $path =
          "admin/structure/taxonomy/manage/$bundle/overview/form-display";
        break;

      case 'block_content':
        $path =
          "admin/structure/block/block-content/manage/$bundle/form-display";
        break;

      case 'comment':
        $path =
          "admin/structure/comment/manage/$bundle/form-display";
        break;

      case 'contact_message':
        $path =
          "admin/structure/contact/manage/$bundle/form-display";
        break;

      default:
        return NULL;

    }

    return $base_url . '/' . $path;

  }

  /**
   * Gets the name of the current edit widget for a paragraph field.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   *
   * @return string
   *   The current widget label.
   */
  private function getCurrentWidget($field_id) {

    [$entity_type, $bundle, $field_name] = explode('.', $field_id);

    $field = $this->entityDisplayRepository
      ->getFormDisplay($entity_type, $bundle)
      ->getComponent($field_name);

    $defs = $this->widgetManager->getDefinitions();

    $current_widget = $field['type'];
    $current_widget_label = $defs[$current_widget]['label']?->render();

    return $current_widget_label;

  }

  /**
   * Formats an option array for getDetailsWidgetOptions.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   * @param \Drupal\field\Entity\FieldConfig $field_config
   *   The field configuration entity.
   *
   * @return array
   *   Formatted option array for the field.
   */
  private function getDetailsWidgetOption($field_id, $field_config) {

    $label = $field_config->get('label');
    $url = $this->getDetailsWidgetOptionUrl($field_id);
    $manage = '';

    if ($url) {
      $manage = $this->t('<a href=":url" target="_blank">Manage Form Display</a>', [':url' => $url]);
    }
    else {
      $manage = $this->t('N / A');
    }

    $current_widget = $this->getCurrentWidget($field_id);

    $option = [
      'paragraph_field' => ['data' => $label, 'class' => 'name-field'],
      'machine_name' => ['data' => $field_id],
      'current_widget' => ['data' => $current_widget],
      'manage' => ['data' => $manage],
    ];

    return $option;

  }

  /**
   * Adds metadata to list of Paragraph fields generated by getFieldConfig.
   *
   * @return array
   *   Array of field configuration entities.
   */
  private function getFieldConfigEntities() {

    $field_config_ids = $this->getFieldConfig();

    $field_config_entities = $this->entityTypeManager
      ->getStorage('field_config')
      ->loadMultiple($field_config_ids);

    return $field_config_entities;

  }

  /**
   * Determines whether or not a bundle currently has any field groups.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return string
   *   'Yes' if field groups exist, 'No' otherwise.
   */
  private function hasFieldGroups($bundle) {

    $groups = field_group_info_groups('node', $bundle, 'form', 'default');

    if (!empty($groups)) {
      return 'Yes';
    }

    return 'No';

  }

  /**
   * Formats an option array for getFieldGroupsOptions.
   *
   * @param string $node_type
   *   The node type machine name.
   * @param string $name
   *   The human-readable name of the content type.
   * @param string $label
   *   The HTML label/link for the content type.
   *
   * @return array
   *   Formatted option array for the content type.
   */
  private function getContentTypeOption($node_type, $name, $label) {

    $has_field_groups = $this->hasFieldGroups($node_type);

    $option = [
      'content_type' => ['data' => $name, 'class' => 'name-field'],
      'machine_name' => ['data' => $node_type],
      'has_field_groups' => ['data' => $has_field_groups],
      'manage' => ['data' => $label],
    ];

    return $option;

  }

  /**
   * Array data function returning info about Settings form structure.
   *
   * Returns information about form container elements.
   *
   * @return array
   *   Array mapping form sections to their corresponding checkbox containers.
   */
  public function getFormStructure() {

    $form_structure = [
      'admin_titles_section' => 'admin_titles_boxes',
      'details_widget_section' => 'details_widget_boxes',
      'field_groups_section' => 'field_groups_boxes',
      'theme_mods_section' => 'theme_mods_boxes',
    ];

    return $form_structure;

  }

  /**
   * Returns a list of Paragraph fields.
   *
   * Only includes fields which can have a
   * Paragraph Details widget added to them.
   *
   * @return array
   *   Array of field configuration IDs.
   */
  public function getFieldConfig() {

    $field_config_ids = $this->entityTypeManager
      ->getStorage('field_config')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('field_type', 'entity_reference_revisions')
      ->condition('settings.handler', 'default:paragraph')
      ->execute();

    return $field_config_ids;

  }

  /**
   * Returns a list of Paragraph Details widget options.
   *
   * @return array
   *   Array of formatted options for paragraph field widgets.
   */
  public function getDetailsWidgetOptions() {

    $field_config_entities = $this->getFieldConfigEntities();

    $options = [];

    foreach ($field_config_entities as $field_id => $field_config) {

      $option = $this->getDetailsWidgetOption($field_id, $field_config);
      $option_id = str_replace('.', '', $field_id);
      $options[$option_id] = $option;

    }

    return $options;

  }

  /**
   * Returns a list of Paragraph types used for Administrative Titles options.
   *
   * @return array
   *   Array of formatted options for paragraph types.
   */
  public function getAdminTitlesOptions() {

    $paragraph_types = $this->getParagraphTypes();
    $options = [];

    foreach ($paragraph_types as $machine_name => $info) {
      $option = $this->getParagraphOption($machine_name, $info);
      $options[$machine_name] = $option;
    }

    return $options;

  }

  /**
   * Returns a list of disabled Field Group checkboxes.
   *
   * @return array
   *   Array of disabled checkbox configurations.
   */
  public function getDisabledFieldGroupBoxes() {

    $types = array_keys($this->getContentTypes());
    $boxes = [];

    foreach ($types as $type) {

      $proceed = $this->fieldGroupManager->proceedWithCreate($type);

      if (!$proceed) {
        $boxes[$type] = ['#disabled' => TRUE];
      }

    }

    return $boxes;

  }

  /**
   * Returns a list of content types used for Field Groups options.
   *
   * @return array
   *   Array of formatted options for content types.
   */
  public function getFieldGroupsOptions() {

    $types = $this->getContentTypes();
    $options = [];

    foreach ($types as $type => $node_type) {

      $name = $node_type->get('name');
      $label = $this->getContentTypeOptionLabel($type);
      $options[$type] = $this->getContentTypeOption($type, $name, $label);

    }

    return $options;

  }

}
