<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\Core\Link;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for paragraph_group helper functions.
 */
class ParagroupHelperService {

  use StringTranslationTrait;

  public function __construct(
    protected ConfigFactory $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected RendererInterface $renderer,
  ) {}

  /**
   * Helper function returning Drupal's main version number.
   *
   * @return string
   *   The major version number.
   */
  public function getDrupalVersion() {

    $version = \Drupal::VERSION;
    $version = explode('.', $version);

    return $version[0];

  }

  /**
   * Validates if the chosen administration theme is permissible.
   *
   * @param string $admin_theme
   *   The admin theme to validate.
   *
   * @return bool
   *   TRUE if the theme is valid, FALSE otherwise.
   */
  public function validateAdminTheme($admin_theme) {

    $compatible_themes = NULL;
    $version = $this->getDrupalVersion();

    if ($version == 10 || $version == 11) {
      $compatible_themes = ['claro', 'gin'];
    }

    if (
      !empty($compatible_themes) &&
      in_array($admin_theme, $compatible_themes)
    ) {
      return TRUE;
    }

    return FALSE;

  }

  /**
   * Adds the theme modification config to the attached drupalSettings.
   *
   * @param array $attached
   *   The attached array to modify.
   *
   * @return bool
   *   TRUE if successful.
   */
  public function addThemeModConfig(array &$attached) {

    $config = $this->configFactory->get('paragraph_group.settings');
    $theme_mods_config = $config->get('paragraph_group.theme_mods_boxes');
    $version = $this->getDrupalVersion();

    $theme_mods_bool = [
      'sc' => boolval($theme_mods_config['sidebar_config']),
      'fwf' => boolval($theme_mods_config['full_width_forms']),
      'ver' => $version,
    ];

    foreach ($theme_mods_bool as $key => $val) {
      if ($val) {
        $attached['drupalSettings']['paragraph_group_' . $key] = $val;
      }
    }

    return TRUE;

  }

  /**
   * Attaches the details widget library to the form attached array.
   *
   * @param array $attached
   *   The attached array to modify.
   */
  public function attachDetailsWidget(array &$attached) {

    // Attach paragraph_group js and css library, and js settings.
    $attached['library'][] = 'paragraph_group/main';
    $attached['drupalSettings']['paragraph_group_details_widget'] = TRUE;

    // Attach theme modification config info as additional js settings.
    $this->addThemeModConfig($attached);

  }

  /**
   * Gets the list of bundles this module is formatting.
   *
   * @return array
   *   Array of bundle machine names.
   */
  public function getFieldGroupBundles() {

    $config = $this->configFactory->get('paragraph_group.settings');
    $boxes_name = 'paragraph_group.field_groups_boxes';
    $bundles = $config->get($boxes_name);

    if (is_array($bundles) && !empty($bundles)) {

      foreach ($bundles as $key => $val) {
        if (!$val) {
          unset($bundles[$key]);
        }
      }

    }

    return $bundles;

  }

  /**
   * Gets the human readable name of a Content Type from its machine name.
   *
   * @param string $bundle
   *   The machine name of the bundle.
   *
   * @return string
   *   The human readable name.
   */
  public function getContentTypeName($bundle) {

    $type = $this->entityTypeManager
      ->getStorage('node_type')
      ->load($bundle);

    return $type->get('name');

  }

  /**
   * Shortcut to get the text for a link.
   *
   * @param string $text
   *   The link text.
   * @param string $route
   *   The route name.
   *
   * @return string
   *   The rendered link markup.
   */
  public function getLink($text, $route) {

    $link = Link::createFromRoute($text, $route);
    $link_render_array = $link->toRenderable();
    $link_render_array['#attributes']['target'] = '_blank';
    $link_markup = $this->renderer->render($link_render_array);

    return $link_markup->__toString();

  }

}
