/// <reference path="./drupal.d.ts" />

function paragroup(

  $: JQueryStatic,
  context: Document | HTMLElement,
  settings: DrupalSettings,

): void {

  const functions = {

    /**
     * Scrolls the browser window to element location.
     * @param el - The jQuery element to scroll to
     */
    scrollToEl(el: JQuery): void {

      // Minus 95px so element scrolls to below Admin Toolbar
      const offsetVal = el.offset()!.top - 95;

      // Use modern scroll behavior instead of jQuery animate
      window.scrollTo({
        top: offsetVal,
        behavior: 'smooth',
      });

    },

    /**
     * Resets the button state to collapsed.
     * @param $button - The jQuery button element to reset
     */
    resetBelow($button: JQuery): void {

      if ($button.hasClass('expanded')) {
        $button.click();
      }
      else {
        $button.click().click();
      }

    },

    /**
     * Expands Open Below buttons in bulk.
     */
    expandBelowBulk(this: HTMLElement): void {

      const $ths = $(this);

      functions.resetBelow($ths);
      $ths.click();

    },

    /**
     * Collapses Open Below buttons in bulk.
     */
    collapseBelowBulk(this: HTMLElement): void {
      functions.resetBelow($(this));
    },

    /**
     * Sets Open Nested buttons to expanded.
     * @param $nestedButtons - The jQuery collection of nested buttons
     */
    setNestedExpanded($nestedButtons: JQuery): void {

      $nestedButtons
        .removeClass('collapsed')
        .addClass('expanded')
        .prop('value', 'Close Nested');

    },

    /**
     * Sets Open Nested buttons to collapsed.
     * @param $nestedButtons - The jQuery collection of nested buttons
     */
    setNestedCollapsed($nestedButtons: JQuery): void {

      $nestedButtons
        .removeClass('expanded')
        .addClass('collapsed')
        .prop('value', 'Open Nested');

    },

    /**
     * Resets all Open Nested buttons to collapsed.
     * @param $button - The jQuery button element to use as context
     */
    resetAllNested($button: JQuery): void {

      const $table = $button.closest('table');
      const $belowButtons = $table.find('.paragroup.toggle-below');
      const $nestedButtons = $table.find('.paragroup.toggle-nested');

      $belowButtons.each(function resetBelowButton(this: HTMLElement) {
        functions.resetBelow($(this));
      });

      functions.setNestedCollapsed($nestedButtons);

    },

    /**
     * Actions to carry out if button is physically clicked.
     * @param event - The click event object
     * @param $ths - The jQuery button element that was clicked
     */
    buttonClicked(event: JQuery.ClickEvent, $ths: JQuery): void {

      if (!(event.originalEvent === undefined)) {

        if ($ths.hasClass('collapsed') && $ths.hasClass('toggle-below')) {
          functions.resetAllNested($ths);
        }

        functions.scrollToEl($ths.siblings('a'));

      }

    },

    /**
     * Determine if a paragraph has sub-paragraphs. Used to prevent Open Nested
     * button being appended to lowest level Paragraphs.
     * @param $paragraph - The jQuery paragraph element to check
     * @return True if paragraph has sub-paragraphs, false otherwise
     */
    paragraphHasSubParagraphs($paragraph: JQuery): boolean {

      let draggableRows = false;

      const $details = $paragraph
        .closest('table')
        .find('> tbody > tr > td > div > details');

      const detailsWidgetSel = '.field--widget-paragraph-group-details-widget';
      const $detailsWidget = $details.find(detailsWidgetSel);

      if ($detailsWidget.length) {

        const $tables = $detailsWidget.find('table');

        if ($tables.length) {

          $tables.each(function processTable(this: HTMLElement) {

            const $table = $(this);
            const $rows = $table.find('> tbody > tr.draggable');

            if ($rows.length > 1) {
              draggableRows = true;
              return false;
            }

          });

        }

      }

      return draggableRows;

    },

    /**
     * Toggles an Open Below button.
     * @param event - The click event object
     */
    toggleBelow(this: HTMLElement, event: JQuery.ClickEvent): void {

      const $ths = $(this);
      event.preventDefault();

      const $details = $ths
        .closest('table')
        .find('> tbody > tr > td > div > details');

      if ($ths.hasClass('collapsed')) {

        $ths.prop('value', 'Close Below');
        $details.prop('open', true);
        $ths.removeClass('collapsed').addClass('expanded');

      }
      else if ($ths.hasClass('expanded')) {

        $ths.prop('value', 'Open Below');
        $details.prop('open', false);
        $ths.removeClass('expanded').addClass('collapsed');

      }

      functions.buttonClicked(event, $ths);

    },

    /**
     * Toggles an Open Nested button.
     * @param event - The click event object
     */
    toggleNested(this: HTMLElement, event: JQuery.ClickEvent): void {

      const $ths = $(this);
      event.preventDefault();

      const $table = $ths.closest('table');
      const $belowButtons = $table.find('.paragroup.toggle-below');
      const $nestedButtons = $table.find('.paragroup.toggle-nested');

      if ($ths.hasClass('collapsed')) {
        functions.setNestedExpanded($nestedButtons);
        $belowButtons.each(functions.expandBelowBulk);
      }
      else if ($ths.hasClass('expanded')) {
        functions.resetAllNested($ths);
      }

      functions.buttonClicked(event, $ths);
    },

    /**
     * Toggles an Open All Paragraphs button.
     * @param event - The click event object
     */
    toggleAll(this: HTMLElement, event: JQuery.ClickEvent): void {

      const $ths = $(this);
      event.preventDefault();

      const $belowButtons = $('.paragroup.toggle-below');
      const $nestedButtons = $('.paragroup.toggle-nested');

      if ($ths.hasClass('collapsed')) {

        $ths.prop('value', 'Close All Paragraphs');
        $ths.removeClass('collapsed').addClass('expanded');
        $belowButtons.each(functions.expandBelowBulk);
        functions.setNestedExpanded($nestedButtons);

      }
      else if ($ths.hasClass('expanded')) {

        $ths.prop('value', 'Open All Paragraphs');
        $ths.removeClass('expanded').addClass('collapsed');
        $belowButtons.each(functions.collapseBelowBulk);
        functions.setNestedCollapsed($nestedButtons);

      }

    },

    /**
     * Get HTML for Open All Paragraphs button.
     * @return The HTML string for the Open All Paragraphs button
     */
    allButtonHtml(): string {

      const value = "'Open All Paragraphs'";
      const classes = "'button button--primary paragroup toggle-all collapsed'";

      const tooltip =
        "'This button opens / closes all the nested items " +
        "for all the paragraphs on this page.'";

      const allButton =
        `<input class=${classes} title=${tooltip} value=${value} readonly>`;

      return allButton;

    },

    /**
     * Checks if the Open All Paragraphs button should be appended.
     * Returns false in the edge case where Field Groups exist but the
     * Paragraphs Field Group tab doesn't exist.
     * @return True if button should be appended, false otherwise
     */
    shouldAppendAllButton(): boolean {

      const editParagroupSel = '#edit-group-paragraphs';
      const fieldGroupTabsWrapperSel = '.field-group-tabs-wrapper';
      const $editParagroup = $(editParagroupSel, context);
      const $fieldGroupTabsWrapper = $(fieldGroupTabsWrapperSel);
      const paragroupExists = $editParagroup.length > 0;
      const fieldGroupsExist = $fieldGroupTabsWrapper.length > 0;

      // Edge case: Field Groups exist but no Paragraphs tab
      if (!paragroupExists && fieldGroupsExist) {
        return false;
      }

      return true;

    },

    /**
     * Finds the appropriate region to append the Open All Paragraphs button.
     * @return jQuery element for the region, or null if not found
     */
    findAllButtonRegion(): JQuery | null {

      const editParagroupSel = '#edit-group-paragraphs';
      const $editParagroup = $(editParagroupSel, context);

      if ($editParagroup.length) {
        return $editParagroup;
      }

      let $regionEl = $('.layout-region-node-main', context);

      if (!$regionEl.length) {
        $regionEl = $('.layout-region--node-main', context);
      }

      if($regionEl.length) {
        return $regionEl;
      }

      return null;

    },

    /**
     * Appends the Open All Paragraphs button to the specified region.
     * @param $regionEl - The jQuery element to append the button to
     */
    appendButtonToRegion($regionEl: JQuery): void {

      const tableSel = 'table.field-multiple-table';

      if ($regionEl.find(tableSel).length) {

        const $buttonAppended = $regionEl.find('.paragroup.toggle-all').length;

        if (!$buttonAppended) {
          const allButton = functions.allButtonHtml();
          $regionEl.prepend(allButton);
        }

      }

    },

    /**
     * Append Open All Paragraphs button to page.
     */
    appendAllButton(): void {

      if(functions.shouldAppendAllButton()) {

        const $regionEl = functions.findAllButtonRegion();

        if ($regionEl) {
          functions.appendButtonToRegion($regionEl);
        }

      }

    },

    /**
     * Get paragraph selector and elements.
     * @return jQuery collection of paragraph elements
     */
    getParagraphEls(): JQuery {

      const paragraphSel =
        '.field--widget-paragraph-group-details-widget ' +
        '> div > div > table > thead > tr > th.field-label';

      return $(paragraphSel, context);

    },

    /**
     * Get HTML for Open Below button.
     * @return The HTML string for the Open Below button
     */
    belowButtonHtml(): string {

      const value = "'Open Below'";
      const classes = "'button button--small paragroup toggle-below collapsed'";

      const tooltip =
        "'This button opens / closes the items immediately below.'";

      const input =
        `<input class=${classes} title=${tooltip} value=${value} readonly>`;

      const belowButton =

        `<div class='paragroup button-wrapper'>
          <span class='buttons'>
            <a class='paragroup scrollto'></a>${input}
          </span>
        </div>`;

      return belowButton;

    },

    /**
     * Append Open Below button to page.
     */
    appendBelowButton(): void {

      const $paragraphEls = functions.getParagraphEls();
      const belowButton = functions.belowButtonHtml();

      $paragraphEls.each(function processParagraphEl(this: HTMLElement) {

        const $element = $(this);

        const $belowAppended =
          $element.find('.paragroup.button-wrapper').length;

        if (!$belowAppended) {
          $element.append(belowButton);
        }

      });

    },

    /**
     * Get HTML for Open Nested button.
     * @return The HTML string for the Open Nested button
     */
    nestedButtonHtml(): string {

      const value = "'Open Nested'";

      const classes =
        "'button button--small paragroup toggle-nested collapsed'";

      const tooltip =
        "'This button opens / closes all the nested items in this paragraph.'";

      const nestedButton = `<input class=${classes} title=${tooltip}
         value=${value} readonly>`;

      return nestedButton;

    },

    /**
     * Append Open Nested button to page.
     */
    appendNestedButton(): void {

      const $paragraphEls = functions.getParagraphEls();
      const nestedButton = functions.nestedButtonHtml();

      $paragraphEls.each(function processNestedParagraphEl(this: HTMLElement) {

        const $element = $(this);

        const $nestedAppended =
          $element.find('.paragroup.toggle-nested').length;

        const hasSubParagraphs =
          functions.paragraphHasSubParagraphs($element);

        if (!$nestedAppended && hasSubParagraphs) {

          $element
            .find('.paragroup.button-wrapper > .buttons')
            .append(nestedButton);

        }

      });

    },

    /**
     * Adds class to html element if not already added
     * @param className - The CSS class name to add
     */
    addHtmlClass(className: string): void {

      const $html = $('html');

      if (!$html.hasClass(className)) {
        $html.addClass(className);
      }

    },

    /**
     * Get the Drupal version number passed in the settings value,
     * and return a class name for this version.
     * @param drupalSettings - The settings object containing version information
     * @return The CSS class name for the version
     */
    getVerClass(drupalSettings: DrupalSettings): string | undefined {

      const version =
        parseInt(drupalSettings.paragraph_group_ver as string, 11);

      let verClass: string | undefined;

      switch (version) {

        case 10:
          verClass = 'pg-d-ten';
          break;

        case 11:
        default:
          verClass = 'pg-d-eleven';
          break;

      }

      return verClass;

    },

    /**
     * Adds theme modifications config from module config page to HTML element
     * so that CSS styling is updated accordingly.
     * @param drupalSettings - The settings object containing theme modification configuration
     */
    themeMods(drupalSettings: DrupalSettings): void {

      const verClass = functions.getVerClass(drupalSettings);

      const configs: Record<string, string | undefined> = {
        paragraph_group_sc: 'pg-sidebar-config',
        paragraph_group_fwf: 'pg-full-width-forms',
        paragraph_group_ver: verClass,
      };

      $.each(configs, function processConfig(key: string, val: string | undefined) {

        if (drupalSettings.hasOwnProperty(key) && val) {
          functions.addHtmlClass(val);
        }

      });

    },

    /**
     * When a user clicks 'remove' in a paragraph, the Paragraphs 'Confirm
     * Removal' button is added via AJAX, and the Detail is loaded in the
     * closed state. This function ensures they are opened.
     */
    confirmRemoval(): void {

      const confirmRemovalSel =
        '.paragraphs-dropbutton-wrapper .confirm-remove';

      const $confirmRemovalEl = $(confirmRemovalSel, context);

      if ($confirmRemovalEl.length) {

        const detailsSel = 'details.js-form-wrapper:not(.field-group-tab)';
        const $details = $confirmRemovalEl.parents(detailsSel);

        $details.prop('open', true);

      }

    },

    /**
     * New Paragraphs are added via AJAX. This function ensures all Details
     * within new Paragraphs content added in this way are open.
     */
    addedParagraph(): void {

      const newContentSel = '.draggable > td > .ajax-new-content';
      const $newContent = $(newContentSel, context);

      if ($newContent.length) {
        const $details = $newContent.find('> div > details');
        $details.prop('open', true);
      }

    },

    /**
     * Handles return to top button click event.
     * @param event - The click event
     */
    handleReturnToTop(event: JQuery.ClickEvent): void {
      event.preventDefault();
      $(window).scrollTop(0);
    },

    /**
     * Appends the 'Return to top' to the Edit page actions at the bottom
     * of the edit page.
     */
    appendReturnToTop(): void {

      const elementSel = '#edit-actions';
      const $elementEl = $(elementSel);

      const elementHtml =
        "<button class='link paragroup return-to-top'>Return to top</button>";

      if (!$('html .paragroup.return-to-top').length) {
        $elementEl.append(elementHtml);
      }

      $('html .paragroup.return-to-top').click(functions.handleReturnToTop);

    },

    /**
     * Initialisation function.
     * @param drupalSettings - The settings object for initialization
     */
    init(drupalSettings: DrupalSettings): void {

      functions.addHtmlClass('paragroup-on');
      functions.themeMods(drupalSettings);
      functions.confirmRemoval();
      functions.addedParagraph();

    },

    /**
     * New content added via AJAX leaves an .ajax-new-content div. Lots
     * of these can become nested if buttons are clicked multiple times. This
     * function removes these divs.
     */
    removeAjaxNewContentDiv(): void {

      const ajaxNewContentSel =
        '.field--widget-paragraph-group-details-widget .ajax-new-content > *';

      $(ajaxNewContentSel).unwrap();

    },

    /**
     * Appends all buttons in this file to the edit page.
     */
    appendButtons(): void {

      functions.removeAjaxNewContentDiv();
      functions.appendAllButton();
      functions.appendBelowButton();
      functions.appendNestedButton();
      functions.appendReturnToTop();

    },

    /**
     * Adds the events for all the buttons, and ensures events are only
     * applied once for the case of new content added via AJAX.
     */
    addToggleButtonEvents(): void {

      const events:
      Record<string, (this: HTMLElement, event: JQuery.ClickEvent) => void> = {

        '.toggle-all': functions.toggleAll,
        '.toggle-below': functions.toggleBelow,
        '.toggle-nested': functions.toggleNested,

      };

      const processEvent: (
        key: string,
        val: (this: HTMLElement, event: JQuery.ClickEvent) => void
      ) => void =

      function processEvent(

        key: string,
        val: (this: HTMLElement, event: JQuery.ClickEvent) => void): void {

        const $buttons = $(key, context);
        const tea = 'toggle-events-added';

        $buttons.each(function processButton(this: HTMLElement) {

          const $button = $(this);

          if (!$button.hasClass(tea)) {
            $button.click(val).addClass(tea);
          }

        });

      };

      // Below registers events and prevents them
      // being registered multiple times.
      $.each(events, processEvent); 

    },

    /**
     * Main function containing all functionality in this file.
     * @param drupalSettings - The settings object for the main function
     */
    main(drupalSettings: DrupalSettings): void {

      functions.init(drupalSettings);
      functions.appendButtons();
      functions.addToggleButtonEvents();

    },

  };

  functions.main(settings);

}


(function initializeParagroupMain($: JQueryStatic): void {

  Drupal.behaviors.paragraph_group = {

    attach(context: Document | HTMLElement, settings: DrupalSettings): void {

      if (settings.hasOwnProperty('paragraph_group_details_widget')) {
        paragroup($, context, settings);
      }

    },

  };

})(jQuery);
