/// <reference path="./drupal.d.ts" />

function paragroup($: JQueryStatic): void {

  const functions = {

    /**
     * Open a Details element.
     * @param id - The ID selector of the Details element to open
     */
    openDetails(id: string): void {

      const $detailsEl = $(id);

      if (!$detailsEl.attr('open')) {
        $detailsEl.attr('open', '');
      }

    },

    /**
     * Clicks the select-all input for a configuration section.
     * @param sectionId - The ID selector of the configuration section
     * @param deselect - Whether to deselect all rows (default: false)
     */
    clickSelectAll(sectionId: string, deselect: boolean = false): void {

      const $selectAll = $(`${sectionId} th.select-all input`);
      const title = $selectAll.attr('title');

      let titleComparison: string | null = null;

      if (!deselect) {
        titleComparison = 'Select all rows in this table';
      }
      else {
        titleComparison = 'Deselect all rows in this table';
      }

      if (title === titleComparison) {
        $selectAll.click();
      }

    },

    /**
     * Set up main sections checkboxes.
     */
    mainSections(): void {

      const sections: string[] = [
        '#edit-details-widget-boxes',
        '#edit-admin-titles-boxes',
        '#edit-field-groups-boxes',
      ];

      $.each(sections, functions.processSection);

    },

    /**
     * Set up theme mods checkboxes.
     */
    themeMods(): void {

      const themeModsId = '#edit-theme-mods-boxes';

      functions.clickSelectAll(themeModsId, true);

      $('#edit-theme-mods-boxes-full-width-forms').prop('checked', true);
      $('#edit-theme-mods-boxes-sidebar-config')
        .removeAttr('disabled')
        .prop('checked', false);

      functions.openDetails(themeModsId);

    },

    /**
     * When "clicking here" link is clicked, sets up recommended configuration.
     */
    recommendedConfig(): void {
      $('#pg-rec-config-link').click(functions.handleRecommendedConfig);
    },

    /**
     * Action for disabling / preventing clicks on sidebar config checkbox,
     * used in function below this one.
     */
    scAction(): void {

      const $sc = $('#edit-theme-mods-boxes-sidebar-config');
      const $fwf = $('#edit-theme-mods-boxes-full-width-forms');

      if ($fwf.prop('checked')) {
        $sc.removeAttr('disabled');
      }
      else {
        $sc.prop('checked', false).attr('disabled', 'disabled');
      }

    },

    /**
     * Prevents 'Move page configuration sidebar to bottom of edit pages'
     * config checkbox from being checked, without 'Full Width Forms'
     * checkbox being checked first.
     */
    sidebarConfig(): void {

      functions.scAction();
      $('#edit-theme-mods-boxes-full-width-forms').change(functions.scAction);

    },

    /**
     * Adds class to html element confirming module has completed initialising.
     * @param className - The CSS class name to add to the html element
     */
    addClassToHtml(className: string): void {

      const $html = $('html');

      if (!$html.hasClass(className)) {
        $html.addClass(className);
      }

    },

    /**
     * Main function containing all functionality in this file.
     */
    main(): void {
      const $sa = $('#edit-theme-mods-boxes th.select-all input');

      if ($sa.length && !$('html').hasClass('pg-settings-complete')) {
        functions.recommendedConfig();
        functions.sidebarConfig();
        functions.addClassToHtml('pg-settings-complete');
      }
    },

    /**
     * Processes each section for recommended configuration.
     * @param index - The index from jQuery each
     * @param id - The section ID
     */
    processSection(index: number, id: string): void {

      functions.clickSelectAll(id);
      functions.openDetails(id);

    },

    /**
     * Handles recommended configuration link click event.
     * @param event - The click event
     */
    handleRecommendedConfig(event: JQuery.ClickEvent): void {

      event.preventDefault();
      functions.mainSections();
      functions.themeMods();

    },

  };

  functions.main();

}

(function initializeParagroupSettings($: JQueryStatic): void {

  Drupal.behaviors.paragraph_group = {

    attach(context: Document | HTMLElement, settings: DrupalSettings): void {

      if (settings.hasOwnProperty('paragraph_group_settings')) {
        paragroup($);
      }

    },

  };

})(jQuery);
