<?php

namespace Drupal\paragraph_group\Hook;

use Drupal\node\NodeForm;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\field\FieldConfigInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\paragraph_group\Paragroup\ParagroupHelperService;
use Drupal\paragraph_group\Paragroup\ParagroupFieldGroupManager;

/**
 * Implements hooks for the Paragraph Group module.
 */
class ParagroupHooks {

  use StringTranslationTrait;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The paragraph group helper service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupHelperService
   */
  protected $helper;

  /**
   * The field group manager service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupFieldGroupManager
   */
  protected $fieldGroupManager;

  public function __construct(
    MessengerInterface $messenger,
    ParagroupHelperService $helper,
    ParagroupFieldGroupManager $field_group_manager,
    TranslationInterface $string_translation,
  ) {

    $this->messenger = $messenger;
    $this->helper = $helper;
    $this->fieldGroupManager = $field_group_manager;
    $this->setStringTranslation($string_translation);

  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(
    string $route_name,
    RouteMatchInterface $route_match,
  ): array|string {

    switch ($route_name) {

      case 'help.page.paragraph_group':
        $output = '';
        $output .= '<h3>' . $this->t('About') . '</h3>';
        $output .= '<p>' . $this->t('The Paragraph Group module enhances the usability of websites utilising the Paragraphs module, and in particular makes highly nested Paragraphs easier to manage and edit for content authors and editors. It also features a settings page to automate the application of the widget to relevant fields, and also automates several other processes for optimising the Paragraph Details widget.') . '</p>';
        $output .= '<h3>' . $this->t('Uses') . '</h3>';
        $output .= '<dl>';
        $output .= '<dt>' . $this->t('Making nested Paragraphs easier to manage and edit') . '</dt>';
        $output .= '<dd>' . $this->t('The Paragraph Details widget is a drop-in replacement for the Paragraphs Legacy (classic) and Paragraphs Stable (AJAX) edit widgets. It utilises the HTML Details element to make nested Paragraphs easier to navigate.') . '</dd>';
        $output .= '<dt>' . $this->t('Applying module functionality in bulk via the settings page') . '</dt>';
        $output .= '<dd>' . $this->t("The Paragraph Group module features a useful settings page for automatically applying all the module's functionality in bulk to all the relevant fields and content items.") . '</dd>';
        $output .= '<dt>' . $this->t('Adding an Administrative Title field to Paragraphs') . '</dt>';
        $output .= '<dd>' . $this->t("An Administrative Title field makes Paragraphs easier to find on the edit page, particularly when using the Paragraph Details widget, where it's content appears as summary text.") . '</dd>';
        $output .= '<dt>' . $this->t('Adding Field Groups to Content Types') . '</dt>';
        $output .= '<dd>' . $this->t("Paragraph Group module also integrates with the Field Group module in order to improve the content editing experience of Paragraphs content, the Paragraph Details widget, and Drupal Content Types more broadly. Paragraph Group module utilises Field Groups to organise groups of fields into Vertical Tabs. In particular, it creates a Paragraphs tab where all Paragraphs fields are placed. This streamlines the usage of the Paragraph Details widget, e.g. by putting the 'Open All Paragraphs' button in this group instead of the top of the page. This makes all the Paragraph fields easier to manage.") . '</dd>';
        $output .= '</dl>';
        $output .= '<h3>' . $this->t('More Information') . '</h3>';
        $output .= '<p>' . $this->t('For more information, please see the module project page at <a href="https://drupal.org/project/paragraph_group" target="_blank">https://drupal.org/project/paragraph_group</a>.') . '</p><br>';

        return $output;

    }

    return '';

  }

  /**
   * Implements hook_form_FORM_ID_alter() for paragroup_config_form.
   */
  #[Hook('form_paragroup_config_form_alter')]
  public function formParagroupConfigFormAlter(
    array &$form,
    FormStateInterface $form_state,
    string $form_id,
  ): void {

    if (!isset($form['#attached']) || !is_array($form['#attached'])) {
      $form['#attached'] = [];
    }

    $attached = &$form['#attached'];

    if (!isset($attached['library']) || !is_array($attached['library'])) {

      $attached['library'] = [];
      $attached['library'][] = 'paragraph_group/settings';

      if (
        !isset($attached['drupalSettings']) ||
        !is_array($attached['drupalSettings'])
      ) {
        $attached['drupalSettings'] = [];
      }

      $attached['drupalSettings']['paragraph_group_settings'] = TRUE;

    }

  }

  /**
   * Implements hook_form_FORM_ID_alter() for system_themes_admin_form.
   */
  #[Hook('form_system_themes_admin_form_alter')]
  public function formSystemThemesAdminFormAlter(
    array &$form,
    FormStateInterface $form_state,
    string $form_id,
  ): void {

    // Use form values instead of raw user input for security.
    $admin_theme = $form_state->getValue('admin_theme');

    if (!empty($admin_theme) && is_string($admin_theme)) {

      // Validate the theme name to prevent path traversal and other attacks.
      if (!$this->helper->validateAdminTheme($admin_theme)) {

        $error_msg = $this->t('The selected theme is not compatible with the Paragraph Group module and so cannot be set as the Administration Theme while the Paragraph Group module is enabled. Please choose a different theme, or disable the Paragraph Group module.');

        $this->messenger->addError($error_msg);

        $form_state->setErrorByName('admin_theme');
        $form_state->setRebuild();

      }

    }

  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(
    array &$form,
    FormStateInterface $form_state,
    string $form_id,
  ): void {

    // Check if form is a node edit form:
    if (class_exists('Drupal\node\NodeForm') && $form_state->getFormObject() instanceof NodeForm) {

      $field_group_bundles = $this->helper->getFieldGroupBundles();

      if ($field_group_bundles === NULL) {
        return;
      }

      foreach ($field_group_bundles as $bundle) {

        $bundle = strtolower($bundle);
        $form_id = strtolower($form_id);

        if ($form_id == 'node_' . $bundle . '_edit_form') {

          /** @var array<string, mixed> $form */
          if (!$this->isLibraryAttached($form)) {

            if (!isset($form['#attached']) || !is_array($form['#attached'])) {
              $form['#attached'] = [];
            }

            /** @var array<string, mixed> $attached_array */
            $attached_array = $form['#attached'];
            $this->helper->attachDetailsWidget($attached_array);

          }

        }

      }

    }

  }

  /**
   * Implements hook_ENTITY_TYPE_presave() for field_config entities.
   */
  #[Hook('field_config_presave')]
  public function fieldConfigPresave(FieldConfigInterface $entity): void {

    if ($this->proceedWithPresaveFieldGroupsReset($entity)) {
      $this->resetFieldGroupsAction($entity);
    }

  }

  /**
   * Implements hook_ENTITY_TYPE_delete() for field_config entities.
   */
  #[Hook('field_config_delete')]
  public function fieldConfigDelete(FieldConfigInterface $entity): void {

    if ($this->proceedWithFieldGroupsReset($entity)) {
      $this->resetFieldGroupsAction($entity);
    }

  }

  /**
   * Checks if 'paragraph_group/main' library is attached to form.
   *
   * @param array<string, mixed> $form
   *   The form array, passed by reference.
   *
   * @return bool
   *   TRUE if library is attached, FALSE otherwise.
   */
  private function isLibraryAttached(&$form): bool {

    foreach ($form as $key => $item) {

      if (strpos($key, 'field_') === 0) {

        if (isset($item['widget']['#attached']['library'])) {

          $library = $item['widget']['#attached']['library'];

          if (
            is_array($library) &&
            in_array('paragraph_group/main', $library)
          ) {
            return TRUE;
          }

        }

      }

    }

    return FALSE;

  }

  /**
   * Determines whether a content type is currently managed.
   *
   * Checks if managed by the Paragraph Group module via the Field Groups
   * section of our configuration page.
   *
   * @param \Drupal\field\FieldConfigInterface $entity
   *   The entity being checked.
   *
   * @return bool
   *   TRUE if bundle is managed, FALSE otherwise.
   */
  private function bundleIsManaged(FieldConfigInterface $entity) {

    $bundle = $entity->getTargetBundle();
    $field_group_bundles = $this->helper->getFieldGroupBundles();

    if ($bundle && $field_group_bundles) {
      if (in_array($bundle, $field_group_bundles)) {
        return TRUE;
      }
    }

    return FALSE;

  }

  /**
   * Determines whether to proceed with Field Groups reset.
   *
   * Will proceed if entity is a node and managed by this module. Called when a
   * field is deleted from an entity.
   *
   * @param \Drupal\field\FieldConfigInterface $entity
   *   The entity being checked.
   *
   * @return bool
   *   TRUE if should proceed with reset, FALSE otherwise.
   */
  private function proceedWithFieldGroupsReset(FieldConfigInterface $entity) {

    $is_node = $entity->getTargetEntityTypeId() === 'node';
    $bundle_is_managed = $this->bundleIsManaged($entity);

    if ($is_node && $bundle_is_managed) {
      return TRUE;
    }

    return FALSE;

  }

  /**
   * Determines whether to proceed with Field Groups reset.
   *
   * Will proceed on the same conditions as function immediately above, if the
   * new field is also complete. Called when a field is added to an entity.
   *
   * @param \Drupal\field\FieldConfigInterface $entity
   *   The entity being checked.
   *
   * @return bool
   *   TRUE if should proceed with reset, FALSE otherwise.
   */
  private function proceedWithPresaveFieldGroupsReset(
    FieldConfigInterface $entity,
  ) {

    $proceed = $this->proceedWithFieldGroupsReset($entity);

    if ($proceed) {
      return TRUE;
    }

    return FALSE;

  }

  /**
   * Prints the reset message when fields are modified.
   *
   * Displays message when fields are added or deleted to/from a Content Type,
   * specifying that the module has reset the Field Groups to cater for the new
   * field configuration.
   *
   * @param string $bundle
   *   The bundle machine name.
   * @param bool $created_or_deleted
   *   TRUE if field groups were created, FALSE if deleted.
   *
   * @return string
   *   The formatted reset message.
   */
  private function getResetMessage($bundle, $created_or_deleted) {

    $msg = NULL;
    $content_type = $this->helper->getContentTypeName($bundle);

    $link_route = 'paragraph_group.form';
    $link_text = $this->t('module configuration page');
    $link = $this->helper->getLink($link_text, $link_route);

    if ($created_or_deleted) {

      $msg_params = [
        '@content_type' => $content_type,
        '@link' => $link,
      ];

      $msg = $this->t('Paragraph Group module has recreated the Field Groups for the <em>@content_type</em> Content Type, as the field configuration for this Content Type was updated, and you currently have this Content Type selected on the Paragraph Group @link under the Field Groups section.', $msg_params);

    }
    else {

      $msg_params = [
        '@content_type' => $content_type,
        '@link' => $link,
      ];

      $msg = $this->t('Paragraph Group module has deleted the Field Groups for the <em>@content_type</em> Content Type, as you currently have this Content Type selected on the Paragraph Group @link under the Field Groups section, and the field configuration for this Content Type has been updated so that it no longer has enough fields for the Paragraph Group module to apply or manage its Field Groups.',
        $msg_params);

    }

    return $msg;

  }

  /**
   * Resets Field Groups for an entity's bundle.
   *
   * @param \Drupal\field\FieldConfigInterface $entity
   *   The entity whose field groups should be reset.
   */
  private function resetFieldGroupsAction(FieldConfigInterface $entity): void {

    $msg = NULL;
    $bundle = $entity->getTargetBundle();

    if ($bundle === NULL) {
      return;
    }

    $created = $this->fieldGroupManager->createFieldGroups($bundle);

    if ($created) {
      $msg = $this->getResetMessage($bundle, TRUE);
    }
    else {
      $this->fieldGroupManager->deleteFieldGroups($bundle);
      $msg = $this->getResetMessage($bundle, FALSE);
    }

    $this->messenger->addStatus($msg);

  }

}
