<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\Core\Render\Markup;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides Batch API operations for managing paragraph widgets and fields.
 *
 * @package Drupal\paragraph_group\Paragroup
 *
 * Contains all functions used by the Drupal Batch API to create and delete
 * Paragraph Details widgets, Administrative Title fields and field groups.
 */
final class ParagroupBatch {

  use StringTranslationTrait;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The form data service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupFormData
   */
  protected $formData;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The field group manager service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupFieldGroupManager
   */
  protected $fieldGroupManager;

  /**
   * The field manager service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupFieldManager
   */
  protected $fieldManager;

  public function __construct(
    TranslationInterface $string_translation,
    MessengerInterface $messenger,
    ParagroupFormData $form_data,
    ParagroupFieldGroupManager $field_group_manager,
    ParagroupFieldManager $field_manager,
  ) {

    $this->stringTranslation = $string_translation;
    $this->messenger = $messenger;
    $this->formData = $form_data;
    $this->fieldGroupManager = $field_group_manager;
    $this->fieldManager = $field_manager;

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {

    return new static(
      $container->get('string_translation'),
      $container->get('messenger'),
      $container->get('paragraph_group.form_data'),
      $container->get('paragraph_group.field_group_manager'),
      $container->get('paragraph_group.field_manager')
    );

  }

  /**
   * General actions for adding or removing a Paragraph Details widget.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   * @param string $type
   *   The widget type to set.
   *
   * @return int
   *   The save result (SAVED_NEW, SAVED_UPDATED, or 0 if nothing was saved).
   */
  private static function addRemoveDetailsWidgetAction($field_id, $type) {

    [$entity_type, $bundle, $field_name] = explode('.', $field_id);

    $field = \Drupal::service('entity_display.repository')
      ->getFormDisplay($entity_type, $bundle)
      ->getComponent($field_name);

    if (!is_array($field)) {
      $field = [];
    }
    /** @var array<string, mixed> $field */
    $field['type'] = $type;

    if (!isset($field['settings']) || !is_array($field['settings'])) {
      $field['settings'] = [];
    }

    $field['settings']['edit_mode'] = 'open';

    $updated = \Drupal::service('entity_display.repository')
      ->getFormDisplay($entity_type, $bundle)
      ->setComponent($field_name, $field)
      ->save();

    return $updated;

  }

  /**
   * Returns function info for each settings form section.
   *
   * @return array<string, array<string, string>>
   *   Array of batch operation data with prefixes and function names.
   */
  private function batchOpsData() {

    $data = [

      'admin_titles_section' => [
        'prefix' => '\Drupal\paragraph_group\Paragroup\ParagroupBatch::',
        'add' => 'createAdminTitleField',
        'remove' => 'deleteAdminTitleField',
      ],

      'details_widget_section' => [
        'prefix' => '\Drupal\paragraph_group\Paragroup\ParagroupBatch::',
        'add' => 'addDetailsWidget',
        'remove' => 'removeDetailsWidget',
      ],

      'field_groups_section' => [
        'prefix' => '\Drupal\paragraph_group\Paragroup\ParagroupBatch::',
        'add' => 'createFieldGroups',
        'remove' => 'deleteFieldGroups',
      ],

    ];

    return $data;

  }

  /**
   * Helper function returning add or remove based on form state analysis.
   *
   * Uses $diff_val generated from analysis of old and new checkbox form state.
   *
   * @param bool $diff_val
   *   The difference value from form state analysis.
   *
   * @return string
   *   Either 'add' or 'remove'.
   */
  private function addRemove($diff_val) {

    if ($diff_val) {
      return 'add';
    }

    return 'remove';

  }

  /**
   * Gets the machine name with dots for a field used in checkboxes.
   *
   * Settings form uses machine name of fields without dots for checkboxes.
   * This function gets the corresponding machine name with the dots.
   *
   * @param string $query_key
   *   The field key without dots.
   *
   * @return string|false
   *   The field config with dots, or FALSE if not found.
   */
  private function getFieldConfigWithDots($query_key) {

    $field_config_ids = $this->formData->getFieldConfig();
    $processed_ids = [];

    foreach ($field_config_ids as $key => $val) {
      $key = str_replace('.', '', $key);
      $processed_ids[$key] = $val;
    }

    if (isset($processed_ids[$query_key])) {
      return $processed_ids[$query_key];
    }

    return FALSE;

  }

  /**
   * Gets batch operations for each settings form section.
   *
   * @param string $section
   *   The form section name.
   * @param array<string, mixed> $boxes_diff
   *   Array of difference values for checkboxes.
   *
   * @return list<array<int, list<bool|string>|string>>
   *   Array of batch operations.
   */
  private function getAdditionalBatchOps($section, $boxes_diff) {

    $ops = [];
    $data = $this->batchOpsData();

    foreach ($boxes_diff as $id => $diff_val) {

      if ($section == 'details_widget_section') {
        $id = $this->getFieldConfigWithDots($id);
      }

      $add_remove = $this->addRemove((bool) $diff_val);
      $prefix = $data[$section]['prefix'];
      $func = $data[$section][$add_remove];

      $ops[] = [$prefix . $func, [$id]];

    }

    return $ops;

  }

  /**
   * Gets translated strings for batch operations.
   *
   * @return array<string, array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>>
   *   Array of translated strings.
   */
  public function getTranslations(): array {

    return [

      'sections' => [
        'atfs' => $this->t('Administrative Title Fields'),
        'pdws' => $this->t('Paragraph Details Widgets'),
        'fgss' => $this->t('Field Group sets'),
      ],

      'actions' => [
        'added' => $this->t('added'),
        'created' => $this->t('created'),
        'deleted' => $this->t('deleted'),
        'removed' => $this->t('removed'),
      ],

      'messages' => [
        'result_message' => $this->t('@result out of @result @section_type @action.'),
        'error_message' => $this->t('An error occured while processing the selected items.'),
      ],

    ];

  }

  /**
   * Returns initial base array for Batch API.
   *
   * @return array<string, mixed>
   *   The base batch array structure.
   */
  private function getBatchBaseData(): array {

    $finished =
      '\Drupal\paragraph_group\Paragroup\ParagroupBatch::finished';

    /**
     * @var array{
     *   title: \Drupal\Core\StringTranslation\TranslatableMarkup,
     *   operations: array<int, mixed>,
     *   progress_message: \Drupal\Core\StringTranslation\TranslatableMarkup,
     *   error_message: \Drupal\Core\StringTranslation\TranslatableMarkup,
     *   finished: string
     * } $batch
     */
    $batch = [
      'title' => $this->t('Updating Paragraph Group module settings'),
      'operations' => [],
      'progress_message' => $this->t('Processed @current out of @total.'),
      'error_message' => $this->t('An error occurred during processing'),
      'finished' => $finished,
    ];

    return $batch;

  }

  /**
   * Analyses checkboxes from old and new form states.
   *
   * Returns the differences for boxes which have been changed.
   *
   * @param array<string, mixed> $old_boxes
   *   The old form state checkbox values.
   * @param array<string, mixed> $new_boxes
   *   The new form state checkbox values.
   *
   * @return array<string, mixed>
   *   Array of differences with TRUE for additions, FALSE for removals.
   */
  private function getBoxesDiff($old_boxes, $new_boxes) {

    $diff = [];

    foreach ($new_boxes as $key => $new_val) {

      if ($key != 'select_all') {

        if (isset($old_boxes[$key])) {

          $old_val = $old_boxes[$key];

          if ($old_val != $new_val) {

            $diff_val = FALSE;

            if (!$old_val && $new_val) {
              $diff_val = TRUE;
            }

            $diff[$key] = $diff_val;

          }

        }
        elseif ($new_val) {
          $diff[$key] = TRUE;
        }

      }

    }

    return $diff;

  }

  /**
   * Array data function returning message info for each function name.
   *
   * @param string $func_name
   *   The function name to get info for.
   * @param array<string, array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>> $translations
   *   Array of translated strings.
   *
   * @return array<int, \Drupal\Core\StringTranslation\TranslatableMarkup>
   *   Array containing section name and action description.
   */
  private static function getSectionInfoFromFuncName(
    $func_name,
    $translations,
  ): array {

    $atfs = $translations['sections']['atfs'];
    $pdws = $translations['sections']['pdws'];
    $fgss = $translations['sections']['fgss'];

    $added   = $translations['actions']['added'];
    $created = $translations['actions']['created'];
    $deleted = $translations['actions']['deleted'];
    $removed = $translations['actions']['removed'];

    $func_names = [
      'createAdminTitleField' => [$atfs, $created],
      'deleteAdminTitleField' => [$atfs, $deleted],
      'addDetailsWidget' => [$pdws, $added],
      'removeDetailsWidget' => [$pdws, $removed],
      'createFieldGroups' => [$fgss, $added],
      'deleteFieldGroups' => [$fgss, $deleted],
    ];

    return $func_names[$func_name];

  }

  /**
   * Aggregates results from batch operations for display.
   *
   * @param array<string, mixed> $results
   *   The results array from batch processing.
   * @param array<string, array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>> $translations
   *   Array of translated strings.
   *
   * @return array<string, array<string, mixed>>
   *   Aggregated results grouped by section and action.
   */
  private static function getAggregatedResults($results, $translations) {

    $aggregated_results = [];

    foreach ($results as $func_name => $result) {

      if ($result) {

        $info = self::getSectionInfoFromFuncName($func_name, $translations);
        $key = $info[0] . '_' . $info[1];

        if (!isset($aggregated_results[$key])) {

          $aggregated_results[$key] = [
            'count' => 0,
            'section' => $info[0],
            'action' => $info[1],
          ];

        }

        if (is_numeric($result)) {
          $aggregated_results[$key]['count'] += (int) $result;
        }

      }

    }

    return $aggregated_results;

  }

  /**
   * Formats aggregated results into message parts.
   *
   * @param array<string, array<string, mixed>> $aggregated_results
   *   The aggregated results from getAggregatedResults().
   * @param array<string, array<string, \Drupal\Core\StringTranslation\TranslatableMarkup>> $translations
   *   Array of translated strings.
   *
   * @return array<int, string>
   *   Array of formatted message parts.
   */
  private static function formatMessageParts($aggregated_results, $translations) {

    $summary_parts = [];

    foreach ($aggregated_results as $data) {

      $result = $data['count'];
      $section_type = $data['section'];
      $action = $data['action'];

      $placeholders = [
        '@result' => $result,
        '@section_type' => $section_type,
        '@action' => $action,
      ];

      $result_message = $translations['messages']['result_message'];
      $message = strtr($result_message, $placeholders);

      $summary_parts[] = $message;

    }

    return $summary_parts;

  }

  /**
   * Finished function used by Batch API.
   *
   * @param bool $success
   *   Whether the batch process was successful.
   * @param array<string, mixed> $results
   *   The results array from batch processing.
   * @param array<int, array<string, mixed>> $operations
   *   The remaining operations.
   */
  public static function finished($success, $results, $operations): void {

    $translations = \Drupal::getContainer()
      ->get('paragraph_group.batch')
      ->getTranslations();

    if ($success && empty($operations)) {

      $results = $results['paragroup'];
      if (!is_array($results)) {
        $results = [];
      }
      /** @var array<string, mixed> $results */
      $aggregated_results = self::getAggregatedResults($results, $translations);

      if (!empty($aggregated_results)) {

        $summary_parts = self::formatMessageParts($aggregated_results, $translations);

        if (!empty($summary_parts)) {

          $combined_message = implode('<br/>', $summary_parts);
          $markup = Markup::create($combined_message);

          \Drupal::messenger()->addStatus($markup);

        }

      }

    }
    else {

      $msg = $translations['messages']['error_message'];
      \Drupal::messenger()->addStatus($msg);

    }

  }

  /**
   * Updates the context variable used by Batch API during processing.
   *
   * Keeps track of the number of times each function has been run.
   *
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   * @param string $func_name
   *   The function name to update the count for.
   */
  private static function updateContext(&$context, $func_name): void {

    if (!isset($context['results']) || !is_array($context['results'])) {
      $context['results'] = [];
    }

    if (!isset($context['results']['paragroup'])) {

      $context['results']['paragroup'] = [
        'createAdminTitleField' => 0,
        'deleteAdminTitleField' => 0,
        'addDetailsWidget' => 0,
        'removeDetailsWidget' => 0,
        'createFieldGroups' => 0,
        'deleteFieldGroups' => 0,
      ];

    }

    if (
      is_array($context['results']['paragroup']) &&
      array_key_exists($func_name, $context['results']['paragroup'])
    ) {

      if (is_numeric($context['results']['paragroup'][$func_name])) {
        $context['results']['paragroup'][$func_name]++;
      }

    }

  }

  /**
   * Function used by Batch API to create Administrative Title fields.
   *
   * @param string $bundle
   *   The paragraph bundle machine name.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   *
   * @return bool
   *   TRUE if field was created successfully.
   */
  public static function createAdminTitleField($bundle, &$context) {

    $container = \Drupal::getContainer();
    $field_manager = $container->get('paragraph_group.field_manager');
    $field_created = $field_manager->createParagraphAdminTitleField($bundle);

    if ($field_created) {
      self::updateContext($context, 'createAdminTitleField');
    }

    return $field_created;

  }

  /**
   * Function used by Batch API to delete Administrative Title fields.
   *
   * @param string $bundle
   *   The paragraph bundle machine name.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   *
   * @return bool
   *   TRUE if field was deleted successfully.
   */
  public static function deleteAdminTitleField($bundle, &$context) {

    $container = \Drupal::getContainer();
    $field_manager = $container->get('paragraph_group.field_manager');
    $field_deleted = $field_manager->deleteParagraphAdminTitleField($bundle);

    if ($field_deleted) {
      self::updateContext($context, 'deleteAdminTitleField');
    }

    return $field_deleted;

  }

  /**
   * General function used by Batch API to add/remove Details widget.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   * @param string $type
   *   The widget type to set.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   * @param string $op
   *   The operation name for context tracking.
   *
   * @return int
   *   The save result (SAVED_NEW, SAVED_UPDATED, or 0 if nothing was saved).
   */
  private static function opDetailsWidget($field_id, $type, &$context, $op) {

    $details_widget_opped =
      self::addRemoveDetailsWidgetAction($field_id, $type);

    if ($details_widget_opped) {
      self::updateContext($context, $op);
    }

    return $details_widget_opped;

  }

  /**
   * Function used by Batch API to add a Paragraph Details widget to a field.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   *
   * @return int
   *   The save result (SAVED_NEW, SAVED_UPDATED, or 0 if nothing was saved).
   */
  public static function addDetailsWidget($field_id, &$context) {

    $op = 'addDetailsWidget';
    $type = 'paragraph_group_details_widget';

    $details_widget_added =
      self::opDetailsWidget($field_id, $type, $context, $op);

    return $details_widget_added;

  }

  /**
   * Removes a Paragraph Details widget from a field.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   *
   * @return int
   *   The save result (SAVED_NEW, SAVED_UPDATED, or 0 if nothing was saved).
   */
  public static function removeDetailsWidget($field_id, &$context) {

    $op = 'removeDetailsWidget';

    /* Consider setting this to the now default Paragraphs (stable)
    AJAX widget, formerly called Paragraphs Experimental widget, now with
    machine name 'paragraphs', as shown in comment below: */
    // $type = 'paragraphs';
    $type = 'entity_reference_paragraphs';

    $details_widget_removed =
      self::opDetailsWidget($field_id, $type, $context, $op);

    return $details_widget_removed;

  }

  /**
   * Function used by Batch API to create Field Groups.
   *
   * @param string $bundle
   *   The paragraph bundle machine name.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   *
   * @return bool
   *   TRUE if field groups were created successfully.
   */
  public static function createFieldGroups($bundle, &$context) {

    $container = \Drupal::getContainer();
    $field_group_manager = $container->get('paragraph_group.field_group_manager');
    $created_groups = $field_group_manager->createFieldGroups($bundle);

    if ($created_groups) {
      self::updateContext($context, 'createFieldGroups');
    }

    return $created_groups;

  }

  /**
   * Function used by Batch API to delete Field Groups.
   *
   * @param string $bundle
   *   The paragraph bundle machine name.
   * @param array<string, mixed> $context
   *   The batch context array, passed by reference.
   *
   * @return bool
   *   TRUE if field groups were deleted successfully.
   */
  public static function deleteFieldGroups($bundle, &$context) {

    $container = \Drupal::getContainer();
    $field_group_manager = $container->get('paragraph_group.field_group_manager');
    $deleted_groups = $field_group_manager->deleteFieldGroups($bundle);

    if ($deleted_groups) {
      self::updateContext($context, 'deleteFieldGroups');
    }

    return $deleted_groups;

  }

  /**
   * Extracts a single box configuration from a config array.
   *
   * @param mixed $config
   *   The configuration array to extract from.
   * @param string $boxes_name
   *   The name of the boxes configuration key.
   *
   * @return array<string, mixed>
   *   The extracted box configuration, or empty array if not found.
   */
  private function extractBoxConfig($config, string $boxes_name): array {

    if (is_array($config) && isset($config[$boxes_name])) {
      $boxes = $config[$boxes_name];
    }
    else {
      $boxes = [];
    }

    if (!is_array($boxes)) {
      $boxes = [];
    }

    /** @var array<string, mixed> $boxes */
    return $boxes;

  }

  /**
   * Returns the final array of data instructing the Batch API on what to do.
   *
   * @param array<int, mixed> $old_and_new_config
   *   Array containing old and new configuration states.
   *
   * @return array<string, mixed>
   *   The complete batch data array for processing.
   */
  public function getBatchData($old_and_new_config) {

    $form_structure = $this->formData->getFormStructure();
    $batch = $this->getBatchBaseData();

    foreach ($form_structure as $section => $boxes) {

      if ($section != 'theme_mods_section') {

        $boxes_name = 'paragraph_group.' . $boxes;

        $old_config = $old_and_new_config[0];
        $new_config = $old_and_new_config[1];

        $old_boxes = $this->extractBoxConfig($old_config, $boxes_name);
        $new_boxes = $this->extractBoxConfig($new_config, $boxes_name);

        /** @var array<string, mixed> $old_boxes */
        /** @var array<string, mixed> $new_boxes */
        $boxes_diff = $this->getBoxesDiff($old_boxes, $new_boxes);

        if (!empty($boxes_diff)) {

          $ops = $this->getAdditionalBatchOps($section, $boxes_diff);

          /** @var array<int, mixed> $operations */
          $operations = $batch['operations'];

          $batch['operations'] = array_merge($operations, $ops);

        }

      }

    }

    return $batch;

  }

}
