<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldTypePluginManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Creates, deletes and manages Field Groups for content type forms.
 *
 * @package Drupal\paragraph_group\Paragroup
 *
 * Creates, deletes and manages Field Groups
 */
final class ParagroupFieldGroupManager {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The field type plugin manager.
   *
   * @var \Drupal\Core\Field\FieldTypePluginManagerInterface
   */
  protected $fieldTypeManager;

  /**
   * The field manager service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupFieldManager
   */
  protected $fieldManager;

  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    FieldTypePluginManagerInterface $field_type_manager,
    ParagroupFieldManager $field_manager,
  ) {

    $this->entityTypeManager = $entity_type_manager;
    $this->fieldTypeManager = $field_type_manager;
    $this->fieldManager = $field_manager;

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {

    return new static(
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.field.field_type'),
      $container->get('paragraph_group.field_manager')
    );

  }

  /**
   * General function to create Field Groups via Field Group module.
   *
   * @param array<string, mixed> $data
   *   Array of field group configuration data.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function createFieldGroup(array $data): bool {

    $new_group = (object) [
      'mode' => 'default',
      'context' => 'form',
      'children' => $data['children'],
      'format_type' => $data['format_type'],
      'group_name' => $data['group_name'],
      'entity_type' => $data['entity_type'],
      'bundle' => $data['bundle'],
      'parent_name' => $data['parent_name'],
      'weight' => $data['weight'],
      'region' => $data['region'],
      'format_settings' => $data['format_settings'],
      'label' => $data['label'],
    ];

    field_group_group_save($new_group);

    return TRUE;

  }

  /**
   * Creates the main tabs container containing each of the Field Groups.
   *
   * @param list<string> $schema
   *   Array of field group names to create.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function createTabsGroup(array $schema, string $bundle): bool {

    $format_settings = [
      'show_empty_fields' => 0,
      'id' => '',
      'classes' => '',
      'direction' => 'vertical',
      'width_breakpoint' => '640',
    ];

    $data = [
      'group_name' => 'group_tabs',
      'entity_type' => 'node',
      'bundle' => $bundle,
      'weight' => 0,
      'label' => 'Tabs',
      'region' => 'content',
      'parent_name' => '',
      'format_type' => 'tabs',
      'format_settings' => $format_settings,
      'children' => $schema,
    ];

    $this->createFieldGroup($data);

    return TRUE;

  }

  /**
   * Saves the data required to create each Field Group under Tabs.
   *
   * @param array<string, mixed> $data
   *   Array of tab configuration data.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function saveTabData(array $data): bool {

    $format_settings = [
      'show_empty_fields' => 0,
      'id' => '',
      'classes' => '',
      'description' => '',
      'formatter' => 'closed',
      'required_fields' => 1,
    ];

    $general_data = [
      'entity_type' => 'node',
      'region' => 'content',
      'parent_name' => 'group_tabs',
      'format_type' => 'tab',
      'format_settings' => $format_settings,
      'children' => [],
    ];

    $data = array_merge($data, $general_data);
    $this->createFieldGroup($data);

    return TRUE;

  }

  /**
   * Adds fields to the field group.
   *
   * @param object $group_obj
   *   The field group object.
   * @param array<int, string> $children
   *   Array of child field names.
   *
   * @return bool
   *   TRUE if successful.
   */
  private static function populateGroup(
    object $group_obj,
    array $children,
  ): bool {

    $group_obj->children = $children;
    field_group_group_save($group_obj);

    return TRUE;

  }

  /**
   * Gets a list of fields for a bundle from the Manage Fields page.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return list<string>
   *   Array of field names including title.
   */
  private function getFieldList(string $bundle): array {

    $config_props = [
      'entity_type' => 'node',
      'bundle' => $bundle,
    ];

    $field_configs = $this->entityTypeManager
      ->getStorage('field_config')
      ->loadByProperties($config_props);

    $fields = array_keys($field_configs);

    foreach ($fields as $i => $field) {

      $field_parts = explode('.', $field);
      $fields[$i] = $field_parts[2];

    }

    $fields = array_merge(['title'], $fields);

    return $fields;

  }

  /**
   * Filters fields and adds full machine name based on entity and bundle.
   *
   * @param array<int, string> $fields
   *   Array of field names.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return array<int, string>
   *   Array of field data with full machine names.
   */
  private static function getFieldData(array $fields, string $bundle): array {

    $field_data = $fields;
    $field_names = ['title', 'body', 'field_body'];

    foreach ($field_names as $field_name) {

      if (($key = array_search($field_name, $field_data)) !== FALSE) {
        unset($field_data[$key]);
      }

    }

    foreach ($field_data as $i => $field) {
      $field = 'node.' . $bundle . '.' . $field;
      $field_data[$i] = $field;
    }

    return $field_data;

  }

  /**
   * Gets list of field types with no UI.
   *
   * @return array<int, string>
   *   Array of field type names that have no UI.
   */
  private function filterFieldList(): array {

    $defs = $this->fieldTypeManager->getDefinitions();

    foreach ($defs as $name => $info) {

      if (is_array($info) && isset($info['no_ui']) && !$info['no_ui']) {
        unset($defs[$name]);
      }

    }

    return array_keys($defs);

  }

  /**
   * Filters fields from field_info which have no UI.
   *
   * @param array{fields: list<string>, field_config: array<string, \Drupal\field\Entity\FieldConfig>} $field_info
   *   Array containing 'fields' and 'field_config' keys.
   *
   * @return array{fields: list<string>, field_config: array<string, \Drupal\field\Entity\FieldConfig>}
   *   Filtered field info with no-UI fields removed.
   */
  private function filterFieldsInfo(array $field_info): array {

    $filter_list = $this->filterFieldList();
    $fields = $field_info['fields'];
    $field_configs = $field_info['field_config'];

    foreach ($field_configs as $machine_name => $field_config) {

      $field_type = $field_config->get('field_type');

      if (in_array($field_type, $filter_list)) {

        $field_parts = explode('.', $machine_name);
        $field_name = $field_parts[2];

        $field_key = array_search($field_name, $fields);

        if ($field_key !== FALSE) {
          unset($fields[$field_key]);
        }

        unset($field_configs[$machine_name]);

      }

    }

    return [
      'fields' => array_values($fields),
      'field_config' => $field_configs,
    ];

  }

  /**
   * Gets info about fields for the analyseContentType function.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return array{fields: list<string>, field_config: array<string, \Drupal\field\Entity\FieldConfig>}
   *   Array with filtered field info.
   */
  private function getFieldsInfo(string $bundle): array {

    $fields = $this->getFieldList($bundle);
    $field_data = $this->getFieldData($fields, $bundle);

    /** @var array<string, \Drupal\field\Entity\FieldConfig> $field_config */
    $field_config = $this->entityTypeManager
      ->getStorage('field_config')
      ->loadMultiple($field_data);

    $field_info = [
      'fields' => $fields,
      'field_config' => $field_config,
    ];

    $filtered_field_info = $this->filterFieldsInfo($field_info);

    return $filtered_field_info;

  }

  /**
   * Gets the type of a field.
   *
   * Returns $field_type if the field isn't an entity reference; otherwise it
   * retrieves the entity reference type from the field settings.
   *
   * @param string $field_name
   *   The field name.
   * @param array{fields: list<string>, field_config: array<string, \Drupal\field\Entity\FieldConfig>} $fields_info
   *   Array of field information.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return string|false
   *   The field type or target type, FALSE if not found.
   */
  private static function getType(
    string $field_name,
    array $fields_info,
    string $bundle,
  ): string|false {

    $field_id = 'node.' . $bundle . '.' . $field_name;
    $field_configs = $fields_info['field_config'];

    if (isset($field_configs[$field_id])) {

      $field_config = $field_configs[$field_id];
      $field_type = $field_config->get('field_type');

      if (
        $field_type !== 'entity_reference' &&
        $field_type !== 'entity_reference_revisions'
      ) {

        if (is_string($field_type)) {
          return $field_type;
        }

      }

      $settings = $field_config->getSettings();
      $target_type = $settings['target_type'] ?? NULL;

      if (is_string($target_type)) {
        return $target_type;
      }

    }

    return FALSE;

  }

  /**
   * Updates the type $info array based on the $type variable.
   *
   * @param array<string, bool> $info
   *   Array of type information flags.
   * @param string|false $type
   *   The field type or target type.
   * @param string $field_name
   *   The field name.
   *
   * @return array<string, bool>
   *   Updated info array with appropriate flags set.
   */
  private static function setTypeInfo(array $info, string|false $type, string $field_name): array {

    if ($type) {

      if ($type == 'paragraph') {
        $info['has-paragraphs'] = TRUE;
      }
      elseif ($type == 'media' || $type == 'file') {
        $info['has-media-uploads'] = TRUE;
      }
      else {
        $info['has-misc'] = TRUE;
      }

    }
    elseif ($field_name != 'title') {
      $info['has-misc'] = TRUE;
    }

    return $info;

  }

  /**
   * Analyses a content type bundle.
   *
   * Returns an array noting the various kinds of fields contained by it.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return array<string, bool>|false
   *   Array of field type flags, FALSE if bundle has too few fields.
   */
  private function analyseContentType(string $bundle): array|false {

    $fields_info = $this->getFieldsInfo($bundle);

    if (count($fields_info['fields']) > 2) {

      $info = [
        'has-body' => FALSE,
        'has-paragraphs' => FALSE,
        'has-media-uploads' => FALSE,
        'has-misc' => FALSE,
      ];

      $filter_list = $this->filterFieldList();

      foreach ($fields_info['fields'] as $field_name) {

        if (
          $field_name == 'body' ||
          $field_name == 'field_body'
        ) {
          $info['has-body'] = TRUE;
        }
        else {

          $type = $this->getType(
            $field_name, $fields_info, $bundle
          );

          if (!in_array($type, $filter_list)) {
            $info = $this->setTypeInfo($info, $type, $field_name);
          }

        }

      }

      return $info;

    }

    return FALSE;

  }

  /**
   * Array data function mapping bundle info to Field Group schema.
   *
   * @return array<string, array<string, bool|string>>
   *   Array containing schema and info-to-schema mapping.
   */
  private static function getSchemaData(): array {

    $schema = [
      'group_main' => TRUE,
      'group_paragraphs' => TRUE,
      'group_media' => TRUE,
      'group_misc' => TRUE,
    ];

    $info_to_schema = [
      'has-body' => 'group_misc',
      'has-paragraphs' => 'group_paragraphs',
      'has-media-uploads' => 'group_media',
      'has-misc' => 'group_misc',
    ];

    $data = [
      'schema' => $schema,
      'info_to_schema' => $info_to_schema,
    ];

    return $data;

  }

  /**
   * Adjusts schema array based on bundle field information.
   *
   * Schema array is initially returned with all keys set to true; this
   * function sets them to false if the bundle has no fields for that group.
   *
   * @param array<string, bool> $field_info
   *   Array of field type flags from analyseContentType.
   *
   * @return list<string>|false
   *   Array of active field group names,
   *   FALSE if only one group would be created.
   */
  private function getFieldGroupSchema(array $field_info): array|false {

    $data = $this->getSchemaData();
    $schema = $data['schema'];

    foreach ($data['info_to_schema'] as $info_item => $schema_item) {

      if (!$field_info[$info_item]) {
        $schema[$schema_item] = FALSE;
      }

    }

    $schema = array_keys(array_filter($schema));

    /** @var list<string> $schema */
    if (count($schema) > 1) {
      return $schema;
    }

    return FALSE;

  }

  /**
   * Returns the custom data required for each of the main Field Group tabs.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return array<string, array<string, string>>
   *   Array of field group configuration data.
   */
  private static function getGroupData(string $bundle): array {

    $data = [

      'group_main' => [
        'group_name' => 'group_main',
        'label' => 'Main',
      ],

      'group_paragraphs' => [
        'group_name' => 'group_paragraphs',
        'label' => 'Paragraphs',
      ],

      'group_media' => [
        'group_name' => 'group_media',
        'label' => 'Media Uploads',
      ],

      'group_misc' => [
        'group_name' => 'group_misc',
        'label' => 'Miscellaneous',
      ],

    ];

    return $data;

  }

  /**
   * Creates all the field group tabs including the main tab container.
   *
   * @param list<string> $schema
   *   Array of field group names to create.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function createGroups(array $schema, string $bundle): bool {

    $this->createTabsGroup($schema, $bundle);
    $data = $this->getGroupData($bundle);
    $i = 1;

    foreach ($schema as $group_name) {
      /** @var string $group_name */

      $group_data = $data[$group_name];

      $group_data['weight'] = $i;
      $group_data['bundle'] = $bundle;

      $this->saveTabData($group_data);

      $i++;

    }

    return TRUE;

  }

  /**
   * Deletes all field groups in a bundle.
   *
   * @param string $entity_type
   *   The entity type machine name.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if successful.
   */
  private static function deleteAllGroupsInBundle(
    string $entity_type,
    string $bundle,
  ): bool {

    $groups =
      field_group_info_groups($entity_type, $bundle, 'form', 'default');

    foreach ($groups as $group_obj) {

      if (is_object($group_obj)) {
        field_group_delete_field_group($group_obj);
      }

    }

    return TRUE;

  }

  /**
   * Prepares a content type for new field groups.
   *
   * Deletes previous field groups and adjusts field weights so fields can be
   * properly added to new groups.
   *
   * @param list<string> $schema
   *   Array of field group names to create.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function prepareContentType(array $schema, string $bundle): bool {

    $count = count($schema) + 1;

    $this->deleteAllGroupsInBundle('node', $bundle);
    $this->fieldManager->prepareFieldWeights('node', $bundle, $count);

    return TRUE;

  }

  /**
   * Gets which group a field should be added to.
   *
   * @param string $bundle
   *   The bundle machine name.
   * @param string $field_name
   *   The field name.
   * @param array<int, string> $schema
   *   Array of available field group names.
   * @param array{fields: list<string>, field_config: array<string, \Drupal\field\Entity\FieldConfig>} $fields_info
   *   Array of field information.
   *
   * @return string
   *   The field group name to add the field to.
   */
  private function getGroupOfField(
    string $bundle,
    string $field_name,
    array $schema,
    array $fields_info,
  ): string {

    $field_names = ['title', 'body', 'field_body'];

    if (in_array($field_name, $field_names)) {
      return 'group_main';
    }

    $type = $this->getType(
      $field_name, $fields_info, $bundle
    );

    if ($type) {

      if ($type == 'paragraph') {
        return 'group_paragraphs';
      }

      if ($type == 'media' || $type == 'file') {
        return 'group_media';
      }

    }

    if (in_array('group_misc', $schema)) {
      return 'group_misc';
    }

    return 'group_main';

  }

  /**
   * Returns an array noting which group each field should be added to.
   *
   * @param list<string> $schema
   *   Array of field group names to create.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return array<string, string>
   *   Array mapping field names to their target groups.
   */
  private function analyseFields(array $schema, string $bundle): array {

    $fields_info = $this->getFieldsInfo($bundle);
    $fields_to_groups = [];

    foreach ($fields_info['fields'] as $field_name) {

      $fields_to_groups[$field_name] =

        $this->getGroupOfField(
          $bundle, $field_name, $schema, $fields_info
        );

    }

    return $fields_to_groups;

  }

  /**
   * Gets the list of children fields for each group.
   *
   * @param list<string> $schema
   *   Array of field group names.
   * @param array<string, string> $fields_to_groups
   *   Array mapping field names to their target groups.
   *
   * @return array<string, array<int, string>>
   *   Array of children fields for each group.
   */
  private static function getChildren(array $schema, array $fields_to_groups): array {

    $children = [];

    foreach ($schema as $group_name) {
      $children[$group_name] = [];
    }

    foreach ($fields_to_groups as $field => $group) {
      $children[$group][] = $field;
    }

    return $children;

  }

  /**
   * Adds children fields to each group.
   *
   * @param list<string> $schema
   *   Array of field group names.
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if successful.
   */
  private function populateGroups(array $schema, string $bundle): bool {

    $fields_to_groups = $this->analyseFields(
      $schema, $bundle
    );

    $children = $this->getChildren(
      $schema, $fields_to_groups
    );

    $groups = field_group_info_groups('node', $bundle, 'form', 'default');

    foreach ($children as $group_name => $children_set) {

      if (isset($groups[$group_name]) && is_object($groups[$group_name])) {

        $this->populateGroup(
          $groups[$group_name], $children_set
        );

      }

    }

    return TRUE;

  }

  /**
   * Analyses bundle and returns whether it is suitable for field groups.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return list<string>|false
   *   Array of field group names if suitable, FALSE otherwise.
   */
  public function proceedWithCreate(string $bundle): array|false {

    $field_info = $this->analyseContentType($bundle);

    if ($field_info) {

      $schema = $this->getFieldGroupSchema($field_info);

      if ($schema) {
        return $schema;
      }

    }

    return FALSE;

  }

  /**
   * Main public function called by ParagroupBatch to create field groups.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if field groups were created successfully.
   */
  public function createFieldGroups(string $bundle): bool {

    $schema = $this->proceedWithCreate($bundle);

    if ($schema) {

      $this->prepareContentType($schema, $bundle);
      $this->createGroups($schema, $bundle);
      $this->populateGroups($schema, $bundle);

      return TRUE;

    }

    return FALSE;

  }

  /**
   * Main public function called by ParagroupBatch to delete field groups.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return bool
   *   TRUE if field groups were deleted successfully.
   */
  public function deleteFieldGroups(string $bundle): bool {

    $deleted_groups = $this->deleteAllGroupsInBundle('node', $bundle);

    return $deleted_groups;

  }

}
