<?php

namespace Drupal\paragraph_group\Paragroup;

use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\WidgetPluginManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\field\Entity\FieldConfig;
use Drupal\paragraphs\ParagraphsTypeInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides form data and checkboxes for the settings configuration form.
 *
 * @package Drupal\paragraph_group\Paragroup
 *
 * Helper class for ParagroupConfigForm; contains functions used to generate
 * checkboxes for this Settings Form.
 */
final class ParagroupFormData {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The widget plugin manager.
   *
   * @var \Drupal\Core\Field\WidgetPluginManager
   */
  protected $widgetManager;

  /**
   * The field group manager service.
   *
   * @var \Drupal\paragraph_group\Paragroup\ParagroupFieldGroupManager
   */
  protected $fieldGroupManager;

  public function __construct(
    TranslationInterface $string_translation,
    EntityTypeManagerInterface $entity_type_manager,
    RequestStack $request_stack,
    EntityDisplayRepositoryInterface $entity_display_repository,
    WidgetPluginManager $widget_manager,
    ParagroupFieldGroupManager $field_group_manager,
  ) {

    $this->setStringTranslation($string_translation);
    $this->entityTypeManager = $entity_type_manager;
    $this->requestStack = $request_stack;
    $this->entityDisplayRepository = $entity_display_repository;
    $this->widgetManager = $widget_manager;
    $this->fieldGroupManager = $field_group_manager;

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {

    return new static(
      $container->get('string_translation'),
      $container->get('entity_type.manager'),
      $container->get('request_stack'),
      $container->get('entity_display.repository'),
      $container->get('plugin.manager.field.widget'),
      $container->get('paragraph_group.field_group_manager')
    );

  }

  /**
   * Gets complete list of Paragraph types.
   *
   * @return array<string, \Drupal\paragraphs\ParagraphsTypeInterface>
   *   Array of paragraph type info, excluding 'from_library' type.
   */
  private function getParagraphTypes(): array {

    /** @var array<string, \Drupal\paragraphs\ParagraphsTypeInterface> $paragraph_types */
    $paragraph_types =
      $this->entityTypeManager->getStorage('paragraphs_type')->loadMultiple();

    if (isset($paragraph_types['from_library'])) {
      unset($paragraph_types['from_library']);
    }

    return $paragraph_types;

  }

  /**
   * Determines whether a Paragraph bundle has an Administrative Title field.
   *
   * @param string $machine_name
   *   The paragraph bundle machine name.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   'Yes' if field exists, 'No' otherwise.
   */
  private function hasAdminTitleField(
    string $machine_name,
  ): TranslatableMarkup {

    $field = FieldConfig::loadByName(
      'paragraph', $machine_name, 'paragroup_admin_title'
    );

    if ($field) {
      return $this->t('Yes');
    }

    return $this->t('No');

  }

  /**
   * Formats an option array for getAdminTitlesOptions.
   *
   * @param string $machine_name
   *   The paragraph bundle machine name.
   * @param \Drupal\paragraphs\ParagraphsTypeInterface $info
   *   The paragraph type entity.
   *
   * @return array<string, array<string, mixed>>|null
   *   Formatted option array for the paragraph type, or null if no request.
   */
  private function getParagraphOption(
    string $machine_name,
    ParagraphsTypeInterface $info,
  ): ?array {

    $request = $this->requestStack->getCurrentRequest();

    if ($request !== NULL) {

      $base_url = $request->getSchemeAndHttpHost();
      $has_field = $this->hasAdminTitleField($machine_name);
      $path = "/admin/structure/paragraphs_type/$machine_name/fields";
      $url = $base_url . $path;

      $manage_fields = $this->t(
        '<a href=":url" target="_blank">Manage Fields</a>',
        [':url' => $url]
      );

      $option = [
        'paragraph_type' => ['data' => $info->label(), 'class' => 'name-field'],
        'machine_name' => ['data' => $machine_name],
        'has_field' => ['data' => $has_field],
        'manage' => ['data' => $manage_fields],
      ];

      return $option;

    }

    return NULL;

  }

  /**
   * Gets the Manage Form Display link for getFieldGroupsOptions.
   *
   * @param string $type
   *   The content type machine name.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|null
   *   HTML link to the form display management page, or null if no request.
   */
  private function getContentTypeOptionLabel(
    string $type,
  ): ?TranslatableMarkup {

    $request = $this->requestStack->getCurrentRequest();

    if ($request !== NULL) {

      $base_url = $request->getSchemeAndHttpHost();
      $path = "/admin/structure/types/manage/$type/form-display";
      $url = $base_url . $path;

      $label = $this->t('<a href=":url" target="_blank">Manage Form Display</a>', [':url' => $url]);

      return $label;

    }

    return NULL;

  }

  /**
   * Returns list of content types.
   *
   * @return array<string, \Drupal\node\NodeTypeInterface>
   *   Array of node type entities.
   */
  private function getContentTypes(): array {

    $types = $this->entityTypeManager
      ->getStorage('node_type')
      ->loadMultiple();

    return $types;

  }

  /**
   * Gets form display url paths for each entity type.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   *
   * @return string|null
   *   The URL to the form display page, or NULL if not supported.
   */
  private function getDetailsWidgetOptionUrl(string $field_id): ?string {

    $request = $this->requestStack->getCurrentRequest();

    if ($request !== NULL) {

      $base_url = $request->getSchemeAndHttpHost();
      $field_parts = explode('.', $field_id);
      $entity_type = $field_parts[0];
      $bundle = $field_parts[1];

      $path = '';

      switch ($entity_type) {

        case 'node':
          $path =
            "admin/structure/types/manage/$bundle/form-display";
          break;

        case 'paragraph':
          $path =
            "admin/structure/paragraphs_type/$bundle/form-display";
          break;

        case 'taxonomy_term':
          $path =
            "admin/structure/taxonomy/manage/$bundle/overview/form-display";
          break;

        case 'block_content':
          $path =
            "admin/structure/block/block-content/manage/$bundle/form-display";
          break;

        case 'comment':
          $path =
            "admin/structure/comment/manage/$bundle/form-display";
          break;

        case 'contact_message':
          $path =
            "admin/structure/contact/manage/$bundle/form-display";
          break;

        default:
          return NULL;

      }

      return $base_url . '/' . $path;

    }

    return NULL;

  }

  /**
   * Gets the name of the current edit widget for a paragraph field.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   *
   * @return string
   *   The current widget label.
   */
  private function getCurrentWidget(string $field_id): string {

    [$entity_type, $bundle, $field_name] = explode('.', $field_id);

    $field = $this->entityDisplayRepository
      ->getFormDisplay($entity_type, $bundle)
      ->getComponent($field_name);

    $defs = $this->widgetManager->getDefinitions();
    $widget_label = 'Unknown';

    if ($field !== NULL && isset($field['type'])) {

      $current_widget = $field['type'];

      if (is_string($current_widget) && isset($defs[$current_widget])) {

        /** @var array{label: \Drupal\Core\StringTranslation\TranslatableMarkup} $widget_def */
        $widget_def = $defs[$current_widget];

        $widget_label = $widget_def['label']->render();

      }

    }

    return $widget_label;

  }

  /**
   * Formats an option array for getDetailsWidgetOptions.
   *
   * @param string $field_id
   *   The field ID in format 'entity_type.bundle.field_name'.
   * @param \Drupal\field\Entity\FieldConfig $field_config
   *   The field configuration entity.
   *
   * @return array<string, array<string, mixed>>
   *   Formatted option array for the field.
   */
  private function getDetailsWidgetOption(
    string $field_id,
    FieldConfig $field_config,
  ): array {

    $label = $field_config->get('label');
    $url = $this->getDetailsWidgetOptionUrl($field_id);
    $manage = '';

    if ($url) {

      $manage = $this->t(
        '<a href=":url" target="_blank">Manage Form Display</a>',
        [':url' => $url]
      );

    }
    else {
      $manage = $this->t('N / A');
    }

    $current_widget = $this->getCurrentWidget($field_id);

    $option = [
      'paragraph_field' => ['data' => $label, 'class' => 'name-field'],
      'machine_name' => ['data' => $field_id],
      'current_widget' => ['data' => $current_widget],
      'manage' => ['data' => $manage],
    ];

    return $option;

  }

  /**
   * Adds metadata to list of Paragraph fields generated by getFieldConfig.
   *
   * @return array<string, \Drupal\field\Entity\FieldConfig>
   *   Array of field configuration entities.
   */
  private function getFieldConfigEntities(): array {

    $field_config_ids = $this->getFieldConfig();

    $field_config_entities = $this->entityTypeManager
      ->getStorage('field_config')
      ->loadMultiple($field_config_ids);

    return $field_config_entities;

  }

  /**
   * Determines whether or not a bundle currently has any field groups.
   *
   * @param string $bundle
   *   The bundle machine name.
   *
   * @return string
   *   'Yes' if field groups exist, 'No' otherwise.
   */
  private function hasFieldGroups(string $bundle): string {

    $groups = field_group_info_groups('node', $bundle, 'form', 'default');

    if (!empty($groups)) {
      return 'Yes';
    }

    return 'No';

  }

  /**
   * Formats an option array for getFieldGroupsOptions.
   *
   * @param string $node_type
   *   The node type machine name.
   * @param string $name
   *   The human-readable name of the content type.
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $label
   *   The HTML label/link for the content type.
   *
   * @return array<string, array<string, mixed>>
   *   Formatted option array for the content type.
   */
  private function getContentTypeOption(
    string $node_type,
    string $name,
    TranslatableMarkup $label,
  ): array {

    $has_field_groups = $this->hasFieldGroups($node_type);

    $option = [
      'content_type' => ['data' => $name, 'class' => 'name-field'],
      'machine_name' => ['data' => $node_type],
      'has_field_groups' => ['data' => $has_field_groups],
      'manage' => ['data' => $label],
    ];

    return $option;

  }

  /**
   * Array data function returning info about Settings form structure.
   *
   * Returns information about form container elements.
   *
   * @return array<string, string>
   *   Array mapping form sections to their corresponding checkbox containers.
   */
  public function getFormStructure(): array {

    $form_structure = [
      'admin_titles_section' => 'admin_titles_boxes',
      'details_widget_section' => 'details_widget_boxes',
      'field_groups_section' => 'field_groups_boxes',
      'theme_mods_section' => 'theme_mods_boxes',
    ];

    return $form_structure;

  }

  /**
   * Returns a list of Paragraph fields.
   *
   * Only includes fields which can have a
   * Paragraph Details widget added to them.
   *
   * @return array<string, string>
   *   Array of field configuration IDs.
   */
  public function getFieldConfig(): array {

    $field_config_ids = $this->entityTypeManager
      ->getStorage('field_config')
      ->getQuery()
      ->condition('field_type', 'entity_reference_revisions')
      ->condition('settings.handler', 'default:paragraph')
      ->execute();

    return $field_config_ids;

  }

  /**
   * Returns a list of Paragraph Details widget options.
   *
   * @return array<string, array<string, array<string, mixed>>>
   *   Array of formatted options for paragraph field widgets.
   */
  public function getDetailsWidgetOptions(): array {

    $field_config_entities = $this->getFieldConfigEntities();

    $options = [];

    foreach ($field_config_entities as $field_id => $field_config) {

      $option = $this->getDetailsWidgetOption($field_id, $field_config);
      $option_id = str_replace('.', '', $field_id);
      $options[$option_id] = $option;

    }

    return $options;

  }

  /**
   * Returns a list of Paragraph types used for Administrative Titles options.
   *
   * @return array<string, array<string, array<string, mixed>>>
   *   Array of formatted options for paragraph types.
   */
  public function getAdminTitlesOptions(): array {

    $paragraph_types = $this->getParagraphTypes();
    $options = [];

    foreach ($paragraph_types as $machine_name => $info) {

      $option = $this->getParagraphOption($machine_name, $info);

      if ($option !== NULL) {
        $options[$machine_name] = $option;
      }

    }

    return $options;

  }

  /**
   * Returns a list of disabled Field Group checkboxes.
   *
   * @return array<string, array<string, bool>>
   *   Array of disabled checkbox configurations.
   */
  public function getDisabledFieldGroupBoxes(): array {

    $types = array_keys($this->getContentTypes());
    $boxes = [];

    foreach ($types as $type) {

      $proceed = $this->fieldGroupManager->proceedWithCreate($type);

      if (!$proceed) {
        $boxes[$type] = ['#disabled' => TRUE];
      }

    }

    return $boxes;

  }

  /**
   * Returns a list of content types used for Field Groups options.
   *
   * @return array<string, array<string, array<string, mixed>>>
   *   Array of formatted options for content types.
   */
  public function getFieldGroupsOptions(): array {

    $types = $this->getContentTypes();
    $options = [];

    foreach ($types as $type => $node_type) {

      $name = $node_type->get('name');

      if (!is_string($name)) {
        $name = '';
      }

      $label = $this->getContentTypeOptionLabel($type);

      if ($label !== NULL) {
        $options[$type] = $this->getContentTypeOption($type, $name, $label);
      }

    }

    return $options;

  }

}
